/**
 * AYU引擎 - 页面检测器
 * 智能识别拼多多页面类型和状态
 */

class AyuPageDetector {
    constructor() {
        this.currentUrl = window.location.href;
        this.currentHostname = window.location.hostname;
        this.pageType = null;
        this.isReady = false;
        
        this.init();
    }

    /**
     * 安全的日志输出方法
     */
    log(level, message, data = null) {
        if (typeof window.AyuLogger !== 'undefined' && window.AyuLogger[level]) {
            window.AyuLogger[level]('PageDetector', message, data);
        } else {
            // 后备日志方案
            const prefix = '🎯 [AYU] [PageDetector]';
            const logMessage = `${prefix} ${message}`;
            data ? console.log(logMessage, data) : console.log(logMessage);
        }
    }

    /**
     * 初始化页面检测
     */
    init() {
        this.detectPageType();
        this.setupUrlChangeListener();
        
        this.log('info', '页面检测完成', {
            url: this.currentUrl,
            hostname: this.currentHostname,
            pageType: this.pageType
        });
    }

    /**
     * 检测页面类型
     */
    detectPageType() {
        const config = window.AyuConfig;
        
        this.log('debug', '开始页面类型检测', {
            hostname: this.currentHostname,
            url: this.currentUrl
        });
        
        // 检查是否为拼多多域名
        const isPdd = config.isPddDomain(this.currentHostname);
        const isMerchant = config.isMerchantDomain(this.currentHostname);
        
        this.log('debug', '域名检测结果', {
            isPddDomain: isPdd,
            isMerchantDomain: isMerchant,
            userDomains: config.pddDomains.user,
            merchantDomains: config.pddDomains.merchant
        });
        
        if (!isPdd) {
            this.pageType = 'unknown';
            this.log('warn', '非拼多多域名', this.currentHostname);
            return;
        }

        // 检测商家后台页面
        if (isMerchant) {
            this.log('debug', '检测为商家后台域名');
            this.detectMerchantPageType();
        } 
        // 检测用户端页面
        else {
            this.log('debug', '检测为用户端域名');
            this.detectUserPageType();
        }

        this.isReady = true;
    }

    /**
     * 检测商家后台页面类型
     */
    detectMerchantPageType() {
        const config = window.AyuConfig;
        const url = this.currentUrl;

        this.log('debug', '检测商家后台页面类型', {
            url: url,
            merchantProductListPaths: config.pageTypes.merchantProductList,
            merchantProductEditPaths: config.pageTypes.merchantProductEdit,
            merchantProductEditAIPaths: config.pageTypes.merchantProductEditAI
        });

        // 商品列表页面 - 精确匹配URL路径
        const pathname = new URL(url).pathname;
        const matchedListPath = config.pageTypes.merchantProductList.find(path => pathname === path);
        if (matchedListPath) {
            this.pageType = 'merchant_product_list';
            this.log('success', '检测到商家后台商品列表页面', {
                matchedPath: matchedListPath,
                pathname: pathname,
                url: url
            });
            return;
        }

        // 商品编辑页面（AI制图功能）- 需要更精确的匹配
        const matchedEditAIPath = config.pageTypes.merchantProductEditAI.find(path => {
            const hasPath = url.includes(path);
            // 额外检查必要的URL参数
            const hasRequiredParams = url.includes('id=') && url.includes('goods_id=');
            return hasPath && hasRequiredParams;
        });
        
        if (matchedEditAIPath) {
            this.pageType = 'merchant_product_edit_ai';
            console.log('🎯 [PageDetector] 检测到商品编辑AI页面', {
                matchedPath: matchedEditAIPath,
                url: url,
                hasRequiredParams: url.includes('id=') && url.includes('goods_id=')
            });
            this.log('success', '检测到商家后台商品编辑页面（AI制图）', {
                matchedPath: matchedEditAIPath,
                url: url
            });
            return;
        }

        // 商品编辑页面
        const matchedEditPath = config.pageTypes.merchantProductEdit.find(path => url.includes(path));
        if (matchedEditPath) {
            this.pageType = 'merchant_product_edit';
            this.log('success', '检测到商家后台商品编辑页面', {
                matchedPath: matchedEditPath,
                url: url
            });
            return;
        }

        // 其他商家后台页面
        this.pageType = 'merchant_other';
        this.log('info', '检测到商家后台其他页面', {
            url: url,
            availableListPaths: config.pageTypes.merchantProductList,
            availableEditPaths: config.pageTypes.merchantProductEdit,
            availableEditAIPaths: config.pageTypes.merchantProductEditAI
        });
    }

    /**
     * 检测用户端页面类型
     */
    detectUserPageType() {
        const config = window.AyuConfig;
        const url = this.currentUrl;

        this.log('debug', '检测用户端页面类型', {
            url: url,
            productDetailPaths: config.pageTypes.productDetail
        });

        // 商品详情页
        const matchedPath = config.pageTypes.productDetail.find(path => url.includes(path));
        if (matchedPath) {
            this.pageType = 'user_product_detail';
            this.log('success', '检测到用户端商品详情页', {
                matchedPath: matchedPath,
                url: url
            });
            return;
        }

        // 其他用户页面
        this.pageType = 'user_other';
        this.log('info', '检测到用户端其他页面', {
            url: url,
            availablePaths: config.pageTypes.productDetail
        });
    }

    /**
     * 监听URL变化
     */
    setupUrlChangeListener() {
        let lastUrl = this.currentUrl;
        
        // 监听前进后退
        window.addEventListener('popstate', () => {
            this.handleUrlChange();
        });

        // 监听DOM变化来检测SPA路由变化
        const observer = new MutationObserver(() => {
            if (location.href !== lastUrl) {
                lastUrl = location.href;
                this.handleUrlChange();
            }
        });

        observer.observe(document, { subtree: true, childList: true });
    }

    /**
     * 处理URL变化
     */
    handleUrlChange() {
        const oldPageType = this.pageType;
        this.currentUrl = window.location.href;
        this.currentHostname = window.location.hostname;
        
        this.log('debug', 'URL变化检测', {
            from: oldPageType,
            newUrl: this.currentUrl
        });

        // 重新检测页面类型
        this.detectPageType();

        // 如果页面类型发生变化，触发事件
        if (oldPageType !== this.pageType) {
            this.dispatchPageChangeEvent(oldPageType, this.pageType);
        }
    }

    /**
     * 触发页面变化事件
     */
    dispatchPageChangeEvent(oldType, newType) {
        const event = new CustomEvent('ayuPageTypeChanged', {
            detail: {
                oldType,
                newType,
                url: this.currentUrl,
                hostname: this.currentHostname
            }
        });
        
        document.dispatchEvent(event);
        
        this.log('info', `页面类型变化: ${oldType} -> ${newType}`);
    }

    /**
     * 获取当前页面类型
     */
    getPageType() {
        return this.pageType;
    }

    /**
     * 检查是否为指定页面类型
     */
    isPageType(type) {
        return this.pageType === type;
    }

    /**
     * 检查是否为商家后台页面
     */
    isMerchantPage() {
        return this.pageType && this.pageType.startsWith('merchant_');
    }

    /**
     * 检查是否为用户端页面
     */
    isUserPage() {
        return this.pageType && this.pageType.startsWith('user_');
    }

    /**
     * 检查是否为商品相关页面
     */
    isProductPage() {
        return this.pageType && this.pageType.includes('product');
    }

    /**
     * 检查页面是否就绪
     */
    isPageReady() {
        return this.isReady;
    }

    /**
     * 等待页面就绪
     */
    waitForReady() {
        return new Promise((resolve) => {
            if (this.isReady) {
                resolve();
                return;
            }

            // 监听页面就绪
            const checkReady = () => {
                if (this.isReady) {
                    resolve();
                } else {
                    setTimeout(checkReady, 100);
                }
            };
            
            checkReady();
        });
    }

    /**
     * 获取页面信息摘要
     */
    getPageInfo() {
        return {
            url: this.currentUrl,
            hostname: this.currentHostname,
            pageType: this.pageType,
            isReady: this.isReady,
            isMerchant: this.isMerchantPage(),
            isUser: this.isUserPage(),
            isProduct: this.isProductPage()
        };
    }
}

// 创建全局页面检测器实例
window.AyuPageDetector = new AyuPageDetector();