/**
 * AYU引擎 - 统一日志管理器
 * 提供分类日志输出和环境控制
 */

class AyuLogger {
    constructor() {
        this.isDevelopment = true; // 开发环境
        this.logLevels = {
            ERROR: 0,
            WARN: 1,
            INFO: 2,
            DEBUG: 3
        };
        this.currentLevel = this.isDevelopment ? this.logLevels.DEBUG : this.logLevels.WARN;
        this.prefix = '🎯 [AYU]';
    }

    /**
     * 错误日志
     */
    error(category, message, data = null) {
        if (this.currentLevel >= this.logLevels.ERROR) {
            const logMessage = `${this.prefix} ❌ [${category}] ${message}`;
            data ? console.error(logMessage, data) : console.error(logMessage);
        }
    }

    /**
     * 警告日志
     */
    warn(category, message, data = null) {
        if (this.currentLevel >= this.logLevels.WARN) {
            const logMessage = `${this.prefix} ⚠️ [${category}] ${message}`;
            data ? console.warn(logMessage, data) : console.warn(logMessage);
        }
    }

    /**
     * 信息日志
     */
    info(category, message, data = null) {
        if (this.currentLevel >= this.logLevels.INFO) {
            const logMessage = `${this.prefix} ℹ️ [${category}] ${message}`;
            data ? console.info(logMessage, data) : console.info(logMessage);
        }
    }

    /**
     * 调试日志
     */
    debug(category, message, data = null) {
        if (this.currentLevel >= this.logLevels.DEBUG) {
            const logMessage = `${this.prefix} 🔍 [${category}] ${message}`;
            data ? console.debug(logMessage, data) : console.debug(logMessage);
        }
    }

    /**
     * 成功日志
     */
    success(category, message, data = null) {
        if (this.currentLevel >= this.logLevels.INFO) {
            const logMessage = `${this.prefix} ✅ [${category}] ${message}`;
            data ? console.log(logMessage, data) : console.log(logMessage);
        }
    }

    /**
     * 系统启动日志
     */
    system(message, data = null) {
        const logMessage = `${this.prefix} 🚀 [SYSTEM] ${message}`;
        data ? console.log(logMessage, data) : console.log(logMessage);
    }

    /**
     * 性能计时开始
     */
    timeStart(category, label) {
        if (this.currentLevel >= this.logLevels.DEBUG) {
            console.time(`${this.prefix} ⏱️ [${category}] ${label}`);
        }
    }

    /**
     * 性能计时结束
     */
    timeEnd(category, label) {
        if (this.currentLevel >= this.logLevels.DEBUG) {
            console.timeEnd(`${this.prefix} ⏱️ [${category}] ${label}`);
        }
    }
}

// 创建全局日志器实例
window.AyuLogger = new AyuLogger();