/**
 * AYU引擎 - 配置管理器
 * 统一管理扩展配置和常量
 */

class AyuConfig {
    constructor() {
        this.version = '2.0.0';
        this.name = '阿欲引擎';
        
        // API配置
        this.api = {
            // 环境配置
            development: {
                baseUrl: 'https://ayu.wanwutime.com',
                debug: true
            },
            production: {
                baseUrl: 'https://ayu.wanwutime.com', // 生产域名，待配置
                debug: false
            },
            
            // 当前环境
            currentEnv: 'development',
            
            // API路径
            paths: {
                USER_LOGIN: '/api/user/login',
                USER_INFO: '/api/user/info',
                FBJ_CONFIG: '/api/fbj/config',
                FBJ_PARAMS: '/api/fbj/params',
                FBJ_GOODS_INFO_ENHANCED: '/api/fbj/goodsInfoEnhanced',
                FBJ_CALCULATE_POWER: '/api/fbj/calculatePower',
                FBJ_JOB_STATUS: '/api/fbj/jobStatus',
                PAY_CREATE_ORDER: '/api/pay/createOrder',
                PAY_QUERY_ORDER: '/api/pay/queryOrder',
                PACKAGE_LIST: '/api/package/list'
            }
        };
        
        // 拼多多相关域名配置
        this.pddDomains = {
            // 用户端域名
            user: [
                'pinduoduo.com',
                'yangkeduo.com',
                'mobile.yangkeduo.com'    // 移动端用户页面
            ],
            // 商家后台域名
            merchant: [
                'mms.pinduoduo.com',      // 商家管理系统
                'mai.pinduoduo.com',      // 商家工作台
                'fuwu.pinduoduo.com',     // 服务商后台
                'seller.pinduoduo.com',   // 商家中心
                'mobile.mai.pinduoduo.com', // 移动端商家工作台
                'app.pinduoduo.com',      // 应用后台
                'pifa.pinduoduo.com'      // 批发后台
            ],
            // 图片CDN域名
            cdn: [
                'pddpic.com',
                'pinduoduo.net'
            ]
        };

        // 页面类型识别配置
        this.pageTypes = {
            // 商品详情页路径特征
            productDetail: [
                '/goods.html',
                '/goods1.html',
                '/goods2.html',
                '/product/',
                '/item/',
                'goods.html',
                'goods1.html',
                'goods2.html'
            ],
            // 商家后台商品管理页面 - 只在特定页面显示
            merchantProductList: [
                '/goods/goods_list'     // 只匹配 mms.pinduoduo.com/goods/goods_list
            ],
            // 商品编辑页面
            merchantProductEdit: [
                '/goods/',
                '/edit'
            ],
            // 商品编辑页面（AI制图功能）
            merchantProductEditAI: [
                '/goods/goods_add/index'
            ]
        };

        // 功能模块配置
        this.features = {
            // 处理类型和对应算力消耗
            processTypes: {
                TITLE_ONLY: { name: '标题优化', power: 1, code: 'title' },
                IMAGE_MAIN: { name: '主图防比价', power: 3, code: 'main_image' },
                IMAGE_SKU: { name: 'SKU图防比价', power: 2, code: 'sku_image' },
                FULL_PROCESS: { name: '完整处理', power: 5, code: 'full' }
            },
            
            // 批量处理配置
            batch: {
                maxConcurrent: 3,        // 最大并发数
                maxQueueSize: 50,        // 最大队列长度
                retryTimes: 2,           // 重试次数
                retryDelay: 3000         // 重试延迟(ms)
            }
        };

        // UI配置
        this.ui = {
            // 主题颜色
            colors: {
                primary: '#ff6600',
                primaryLight: '#ff8533',
                success: '#52c41a',
                warning: '#faad14',
                error: '#ff4d4f',
                info: '#1890ff'
            },
            
            // 动画配置
            animation: {
                duration: 300,
                easing: 'ease-in-out'
            },
            
            // Z-index层级
            zIndex: {
                toolbar: 10000,
                dialog: 10001,
                loading: 10002,
                tooltip: 10003
            }
        };

    }

    /**
     * 检查是否为拼多多域名
     */
    isPddDomain(hostname) {
        const allDomains = [
            ...this.pddDomains.user,
            ...this.pddDomains.merchant,
            ...this.pddDomains.cdn
        ];
        return allDomains.some(domain => hostname === domain || hostname.endsWith('.' + domain));
    }

    /**
     * 检查是否为商家后台域名
     */
    isMerchantDomain(hostname) {
        return this.pddDomains.merchant.some(domain => hostname === domain || hostname.endsWith('.' + domain));
    }

    /**
     * 获取当前环境的API基础URL
     */
    getApiBaseUrl() {
        return this.api[this.api.currentEnv].baseUrl;
    }

    /**
     * 生成完整的API URL
     */
    getApiUrl(path = '') {
        const baseUrl = this.getApiBaseUrl();
        if (path.startsWith('/')) {
            return baseUrl + path;
        }
        return baseUrl + '/' + path;
    }

    /**
     * 获取完整的API URL（推荐使用）
     */
    api(pathKey, params = {}) {
        if (!this.api || !this.api.paths) {
            console.error(`[AYU Config] API配置未初始化`);
            return '';
        }
        const path = this.api.paths[pathKey];
        if (!path) {
            console.error(`[AYU Config] API路径 ${pathKey} 未找到`);
            return '';
        }
        
        let url = this.getApiUrl(path);
        
        // 处理URL参数
        if (Object.keys(params).length > 0) {
            const urlParams = new URLSearchParams(params);
            url += (url.includes('?') ? '&' : '?') + urlParams.toString();
        }
        
        return url;
    }

    /**
     * 环境检测和自动切换
     */
    autoDetectEnvironment() {
        // 扩展程序可以通过manifest版本或其他方式检测环境
        const manifest = chrome?.runtime?.getManifest?.();
        if (manifest && manifest.version && manifest.version.includes('dev')) {
            this.api.currentEnv = 'development';
        } else {
            // 也可以通过当前页面域名判断
            const hostname = window.location?.hostname || '';
            if (hostname.includes('localhost') || hostname.includes('127.0.0.1') || hostname.includes('twww')) {
                this.api.currentEnv = 'development';
            } else {
                this.api.currentEnv = 'production';
            }
        }
        
        if (this.api.development.debug && this.api.currentEnv === 'development') {
            console.log(`[AYU Config] 检测到环境: ${this.api.currentEnv}, API地址: ${this.getApiBaseUrl()}`);
        }
    }

    /**
     * 手动设置环境
     */
    setEnvironment(env) {
        if (env === 'development' || env === 'production') {
            this.api.currentEnv = env;
            if (this.api.development.debug) {
                console.log(`[AYU Config] 手动设置环境: ${this.api.currentEnv}, API地址: ${this.getApiBaseUrl()}`);
            }
        } else {
            console.error(`[AYU Config] 无效的环境设置: ${env}`);
        }
    }
}

// 创建全局配置实例
window.AyuConfig = new AyuConfig();

// 自动检测环境
window.AyuConfig.autoDetectEnvironment();