/**
 * AYU引擎 Service Worker
 * 专为PinDuoDuo商家后台防比价功能设计
 */

// 启动Service Worker
console.log('🚀 [Service Worker] 启动AYU引擎Service Worker...');

// 使用 declarativeNetRequest API 进行网络拦截
if (chrome.declarativeNetRequest) {
    console.log('🔧 [Service Worker] 设置声明式网络拦截器...');
    
    // 生成拦截数据
    function generateInterceptData(type) {
        if (type === 'getScene') {
            return {
              "success": true,
              "errorCode": 1000000,
              "errorMsg": null,
              "result": [
                  {
                      "name": "纯色摄影棚",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/1c993dc7-0ca9-450e-aff5-601bfbe596a7.jpg",
                      "frontType": "1",
                      "frontTypeName": "室内",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          18637,
                          8509,
                          8509,
                          18637,
                          9324,
                          9313,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "山地",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/92b61dc6-ecff-4f32-a676-a4d514c38544.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8538,
                          8509,
                          11688,
                          11689,
                          11683
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "足球场",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/322e2248-65f3-47b9-b6b8-268e0874b952.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8509,
                          11688
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "中式客厅",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/1d8496bc-ce15-4ef3-849d-05e312615fc2.jpg",
                      "frontType": "1",
                      "frontTypeName": "室内",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8538
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "沙发",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/a43985e6-16f1-44a8-9013-db2f7634186b.jpg",
                      "frontType": "1",
                      "frontTypeName": "室内",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "雪地",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/6b4cca75-11cb-45b9-9d8e-8dee87694d03.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          18601,
                          18637,
                          8509,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "砂石",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/bcf91d74-3417-4287-9d6d-c47c5b397150.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8509,
                          11688
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "有水路面",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/7f783eaf-7c4d-496b-81eb-9e2bf694e6ba.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8509,
                          11688,
                          11689,
                          11683
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "湖畔",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/97d7c9a0-1347-485b-bf8f-9023e5c33423.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8509,
                          11688
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "田园风光",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/dc18ffbe-7d2e-408e-89d2-a99c75479759.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8509,
                          11688,
                          11689,
                          11683
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "展台",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/391b8d54-cefa-4941-bb19-fc9950e8e333.jpg",
                      "frontType": "1",
                      "frontTypeName": "室内",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8509,
                          11688,
                          17455,
                          17412
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "旷野",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/57b04d75-e1c4-4d8e-b0f9-ad90d042bed9.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8509,
                          11688,
                          11689,
                          11683
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "篮球场",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/55852ce1-f2df-4072-83c6-23683fa9cf8c.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8509,
                          11688
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "城市公园",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/dea622cc-af5c-4f5b-bfac-1d0cc08979cf.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8538,
                          15356,
                          18601,
                          18637,
                          8509,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "海岸",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/effe50b9-8826-4787-a735-6683c286263d.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8509,
                          11688,
                          11689,
                          11683
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "园林",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/c45988ba-b3d7-4993-876e-429c99924478.jpg",
                      "frontType": "2",
                      "frontTypeName": "室外",
                      "type": 0,
                      "typeList": [
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8508,
                          14933,
                          8538,
                          8509,
                          11688,
                          11689,
                          11683
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "正红",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/c8daac17-b156-4cc0-a63d-ff745a57b60f.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "红",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/f57a7bc9-b36e-4dcc-8e41-910307249696.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "粉",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/b267d17b-863c-4a96-850d-16a63d8e0735.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "橙",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/83023098-94e2-4fa7-afec-ca08c6e621d2.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "黄",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/61a985f9-47bb-402c-a6cf-6eeea0fe32c9.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "绿",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/8bf1d1ec-ee95-41f1-873e-1ae834d6fd24.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "青",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/6096eca0-f3af-48ee-95fe-7a7be7477ab2.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "蓝",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/a3fd7232-ef5c-402c-be99-231664595c45.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "紫",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/5712b4eb-ba60-41c6-9f65-4ae53794460d.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "灰",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/b65fc3a5-252c-4fcb-80ff-51d519b6e6a3.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "黑",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/b2d40106-51d4-40f8-aced-e9290e1e6349.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  },
                  {
                      "name": "白",
                      "url": "https://img.pddpic.com/aid-image/ai_b_tool/dae7d4fd-d6aa-40a9-bc51-48c214f26ff7.jpg",
                      "frontType": "3",
                      "frontTypeName": "纯色",
                      "type": 0,
                      "typeList": [
                          "人台图",
                          "真人图",
                          "平铺图",
                          "实物图"
                      ],
                      "bokehImagePercentage": 0.0,
                      "cateIdList": [
                          8439,
                          8583,
                          11687,
                          11684,
                          11685,
                          239,
                          14966,
                          8508,
                          8669,
                          14933,
                          15356,
                          8538,
                          18601,
                          9313,
                          18637,
                          8509,
                          9324,
                          9319,
                          11688,
                          18482,
                          18574,
                          11689,
                          11683,
                          17455,
                          5955,
                          17412,
                          6758,
                          9323,
                          16548,
                          16676,
                          6536,
                          6630,
                          2629,
                          9317,
                          9316,
                          9322,
                          6883,
                          15083,
                          6586,
                          6785,
                          6398,
                          15543,
                          24699,
                          18814,
                          20645,
                          14740,
                          19298,
                          14697,
                          21417,
                          8634,
                          7639,
                          13177,
                          7629,
                          5752,
                          2933
                      ],
                      "cate1TagType": null,
                      "cate1TagName": null,
                      "cate2TagType": null,
                      "cate2TagName": null,
                      "cate1IdList": null
                  }
              ]
          };
        }
    }
    
    // 定义拦截规则
    const interceptRules = [
        {
            "id": 1,
            "priority": 1,
            "action": {
                "type": "redirect",
                "redirect": {
                    "url": "data:application/json;charset=utf-8," + 
                          encodeURIComponent(JSON.stringify(generateInterceptData('getScene')))
                }
            },
            "condition": {
                "urlFilter": "*://aid.pinduoduo.com/api/banner-intelligent/smartTool/getScene*",
                "resourceTypes": ["xmlhttprequest"]
            }
        }
    ];
    
    // 设置拦截规则
    chrome.declarativeNetRequest.updateDynamicRules({
        removeRuleIds: [1, 2], // 先删除可能存在的旧规则
        addRules: interceptRules
    }).then(() => {
        console.log('🔧 [Service Worker] ✅ 声明式拦截规则设置成功');
        console.log('🔧 [Service Worker] 规则数量:', interceptRules.length);
        
        // 验证规则是否设置成功
        chrome.declarativeNetRequest.getDynamicRules().then(rules => {
            console.log('🔧 [Service Worker] 当前活动规则:', rules);
        });
        
    }).catch(error => {
        console.error('🔧 [Service Worker] ❌ 声明式拦截规则设置失败:', error);
    });
    
} else {
    console.log('🔧 [Service Worker] declarativeNetRequest API 不可用');
}

/**
 * 注入脚本处理函数
 * 
 * @param {Object} command - 命令对象，包含注入相关的参数
 * @param {Object} sender - 发送方信息，通常包含tab和frame信息
 * @param {Function} callback - 执行完成后的回调函数
 */
function ayuInjectHandler(command, sender, callback) {
  // 默认在主世界上下文中执行脚本
  var worldContext = "MAIN";
  
  // 如果命令指定了执行环境，则使用指定的环境
  if (command.world) {
    worldContext = command.world;
  }
  
  // 如果是类型2，表示直接注入文件而非执行函数
  if (2 == command.type) {
    const executionData = ayuFillIframeData(command, sender, {
      world: worldContext,
      files: command.fileNames // 要注入的文件列表
    });
    
    // 执行脚本注入并返回结果
    return void chrome.scripting.executeScript(executionData).then((result => {
      callback(result);
    }));
  }

  // 处理普通脚本注入（执行函数）
  var executionData;
  var scriptArgs = [command.params]; // 默认参数
  
  // 如果命令中指定了参数数组，则使用该数组
  if (command.args) {
    scriptArgs = command.args;
  }
  
  // 准备执行数据
  executionData = ayuFillIframeData(command, sender, {
    function: injectScript, // 要执行的函数
    args: scriptArgs,      // 函数参数
    world: worldContext     // 执行环境
  });
  
  // 执行脚本并处理结果
  chrome.scripting.executeScript(executionData).then((result => {
    callback(result);
  }));
}

/**
 * 在页面上下文中注入并执行脚本
 * 
 * @param {Object} params - 注入参数，包含类型和值
 * @return {*} - 如果是eval类型，返回eval执行结果，否则返回undefined
 */
function injectScript(params) {
  if (params) {
    // 如果是eval类型，则执行脚本内容，否则返回undefined
    return "eval" == params.type ? eval(params.value) : void 0;
  }
}

/**
 * 辅助函数：填充iframe ID到执行数据中
 * 用于确定脚本应该在哪个iframe中执行
 * 
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Object} executionData - 执行数据对象
 * @return {Object} - 处理后的执行数据对象
 */
function ayuFillIframeData(command, sender, executionData) {
  // 获取iframe ID，优先使用命令中指定的ID
  var iframeId = command.myIframeId;
  
  // 如果命令中没有指定，但发送方有frameId，则使用发送方的frameId
  if (!iframeId && sender.frameId) {
    iframeId = sender.frameId;
  }
  
  // 如果有iframe ID，则创建frameIds数组
  var frameIds = null;
  if (iframeId) {
    frameIds = [iframeId];
  }
  
  // 获取标签页ID
  var tabId = command.tabId;
  if (!tabId) {
    tabId = sender.tab.id;
  }
  
  // 设置目标信息
  executionData.target = {
    tabId: tabId
  };
  
  // 如果有frameIds，则添加到目标信息中
  if (frameIds) {
    executionData.target.frameIds = frameIds;
  }
  
  // 如果命令中指定了执行目标，则使用命令中的执行目标
  if (command.execteTarget) {
    executionData.target = command.execteTarget;
  }
  
  return executionData;
}

/**
 * 注入模块导出
 */
const inject = {
  Lister: ayuInjectHandler
};

/**
 * HTTP 请求头处理模块
 */

// 自定义头信息前缀，用于识别需要特殊处理的头信息
// 如果请求头名称以此前缀开头，会被转换为真实的HTTP请求头
// 例如：'ayu-header-referer' 将被转换为 'referer' 头
 var headerPre = "ayu-header-";

/**
 * 使用正则表达式从字符串中提取匹配模式
 * 
 * @param {string} inputString - 输入字符串
 * @param {RegExp} pattern - 正则表达式模式
 * @return {string|boolean} - 如果匹配成功则返回第一个捕获组，否则返回false
 */
function RegOne(inputString, pattern) {
  if (!inputString) return false;
  
  var matches = pattern.exec(inputString);
  // 检查是否匹配成功且有捕获组
  return !!(matches && matches.length > 1) && matches[1];
}

/**
 * 更新请求规则，用于修改HTTP请求头
 * 允许扩展修改发送到服务器的HTTP请求头
 * 
 * @param {Object} requestOptions - 请求选项对象
 * @return {Object} - 处理后的头信息对象
 */
async function UpdateRules(requestOptions) {
  // 获取头信息，支持 headers 或 header 属性
  var headerObj = requestOptions.headers || requestOptions.header;
  var normalHeaders = {}; // 存储常规头信息
  var hasSpecialHeaders = false; // 是否有需要特殊处理的头
  var domainPart = ""; // 域名部分，用于过滤规则
  
  // 从 URL 中提取域名部分
  if (requestOptions.url) {
    domainPart = RegOne(requestOptions.url, /\/\/([^\/]+)/);
  }
  
  // 保存需要特殊处理的头信息
  var specialHeadersConfig = [];
  
  // 遍历全部头信息，分离普通头和需要特殊处理的头
  for (var headerName in headerObj) {
    var headerValue = headerObj[headerName];
    
    // 将头名转为小写并检查是否包含特殊前缀
    if ((headerName = String(headerName).toLowerCase()).includes(headerPre)) {
      hasSpecialHeaders = true;
      // 移除前缀得到真实的头名
      var realHeaderName = headerName.replace(headerPre, "");
      
      // 添加到特殊头配置中
      specialHeadersConfig.push({
        header: realHeaderName,
        operation: "set", // 设置操作
        value: headerValue
      });
    } else {
      // 普通头直接保存
      normalHeaders[headerName] = headerValue;
    }
  }
  
  // 如果没有特殊头，则直接返回普通头
  if (!hasSpecialHeaders) return normalHeaders;
  
  // 准备请求头修改列表
  var requestHeadersList = [];
  
  // 默认请求条件
  var requestCondition = {
    urlFilter: domainPart,
    resourceTypes: ["xmlhttprequest"] // 默认只对XHR请求生效
  };
  
  // 将特殊头配置添加到请求头列表
  specialHeadersConfig.map((item => {
    requestHeadersList.push(item);
  }));
  
  // 如果请求选项中指定了请求头，则使用指定的
  if (requestOptions.requestHeaders) {
    requestHeadersList = requestOptions.requestHeaders;
  }
  
  // 如果请求选项中指定了条件，则使用指定的
  if (requestOptions.condition) {
    requestCondition = requestOptions.condition;
  }
  
  // 默认的规则配置
  var rulesToAdd = [{
    id: 14, // 规则ID
    priority: 1, // 规则优先级
    action: {
      type: "modifyHeaders", // 修改头类型的动作
      requestHeaders: requestHeadersList // 要修改的请求头
    },
    condition: requestCondition // 规则生效条件
  }];
  
  // 默认要移除的规则ID
  var rulesToRemove = [14];
  
  // 如果请求选项中指定了要添加的规则，则使用指定的
  if (requestOptions.addRules && requestOptions.addRules.length > 0) {
    rulesToAdd = requestOptions.addRules;
  }
  
  // 如果请求选项中指定了要移除的规则ID，则使用指定的
  if (requestOptions.removeRuleIds && requestOptions.removeRuleIds.length > 0) {
    rulesToRemove = requestOptions.removeRuleIds;
  }
  
  // 更新浏览器的声明式网络请求规则
  await chrome.declarativeNetRequest.updateSessionRules({
    removeRuleIds: rulesToRemove,
    addRules: rulesToAdd
  });
  
  // 返回处理后的普通头信息
  return normalHeaders;
}

/**
 * 清除所有的声明式网络请求规则
 * 并设置一个默认的规则
 */
async function ClearRules() {
  // 移除现有规则并添加一个默认规则
  await chrome.declarativeNetRequest.updateSessionRules({
    removeRuleIds: [14, 999], // 移除ID为14和999的规则
    addRules: [{
      action: {
        requestHeaders: [{
          header: "referer",
          operation: "set",
          value: "https://aaabg.com"
        }, {
          header: "origin",
          operation: "set",
          value: "https://aaabg.com"
        }],
        type: "modifyHeaders"
      },
      condition: {
        resourceTypes: ["ping"], // 只对ping类型请求生效
        urlFilter: "aaabg.com"   // 只对包含aaabg.com的URL生效
      },
      id: 999,       // 规则ID
      priority: 1    // 规则优先级
    }]
  });
}

/**
 * HTTP规则处理器
 * 
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 */
async function ayuHttpRuleHandler(command, sender, callback) {
  // 更新规则并调用回调
  await UpdateRules(command);
  callback("");
}

/**
 * HTTP规则模块导出
 */
const http_rule = {
  UpdateRules: UpdateRules,
  Lister: ayuHttpRuleHandler,
  ClearRules: ClearRules
};

/**
 * 网络请求模块
 */

/**
 * 处理带有头信息修改的fetch请求
 * 该函数基于浏览器的fetch API封装，增加了对头信息的特殊处理
 * 
 * @param {Object} requestConfig - 请求配置对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 */
async function ayuFetchHandler(requestConfig, sender, callback) {
  // 初始化请求参数，首先处理头信息
  var fetchParams = {
    headers: await http_rule.UpdateRules(requestConfig)
  };
  
  // 如果配置了方法，添加到参数中
  if (requestConfig.method) {
    fetchParams.method = requestConfig.method;
  }
  
  // 如果配置了数据，添加到请求体中
  if (requestConfig.data) {
    fetchParams.body = requestConfig.data;
  }
  
  // 如果直接指定了完整的fetch参数，使用指定的参数
  if (requestConfig.fetchParams) {
    fetchParams = requestConfig.fetchParams;
  }
  
  // 如果是base64类型的请求，使用特殊处理
  if (requestConfig.type && "base64" == requestConfig.type.toLowerCase()) {
    return GetBase64(requestConfig.url, callback);
  }
  
  // 执行fetch请求
  fetch(requestConfig.url, fetchParams)
    .then(async (response) => {
      if (response.ok) {
        // 如果指定了特殊的解码类型，使用TextDecoder处理
        if (requestConfig.textDecoderType) {
          const decoder = new TextDecoder(requestConfig.textDecoderType);
          var buffer = await response.arrayBuffer();
          return decoder.decode(new Uint8Array(buffer));
        }
        // 默认以文本形式返回
        return response.text();
      }
      // 如果响应不成功，抛出错误
      throw await response.text();
    })
    .then((responseText) => {
      // 尝试将响应解析为JSON
      let parsedJson = null;
      try {
        parsedJson = JSON.parse(responseText);
      } catch (error) {
        // 如果解析失败，保持parsedJson为null
      }
      
      // 构造成功响应对象
      var successResponse = {
        result: parsedJson || responseText, // 优先使用解析后的JSON，如果解析失败则用原始文本
        resultContent: responseText,       // 原始响应文本
        success: true                      // 标记为成功
      };
      
      // 如果设置了不需要清除规则，直接返回结果
      if (requestConfig.isNotNeedClearRules) {
        return callback(successResponse);
      }
      
      // 清除规则后返回结果
      http_rule.ClearRules().then(() => {
        callback(successResponse);
      });
    })
    .catch((error) => {
      // 构造错误响应对象
      var errorResponse = {
        result: error,
        success: false
      };
      
      // 如果设置了不需要清除规则，直接返回错误
      if (requestConfig.isNotNeedClearRules) {
        return callback(errorResponse);
      }
      
      // 清除规则后返回错误
      http_rule.ClearRules().then(() => {
        callback(errorResponse);
      });
    });
}

/**
 * 将URL转换为base64数据
 * 获取远程资源并转换为base64编码的数据内容
 * 
 * @param {string} url - 要获取的资源URL
 * @param {Function} callback - 完成后的回调函数，将被传入base64数据
 */
function GetBase64(url, callback) {
  fetch(url)
    .then((response) => response.blob()) // 转换为blob对象
    .then((blob) => {
      // 创建FileReader实例读取blob数据
      var reader = new FileReader();
      // 获取blob类型（此处件不做任何处理，原代码可能有第二个b.type的格式错误）
      blob.type && blob.type;
      
      // 设置onload事件处理程序
      reader.onload = function() {
        return function(event) {
          // 将结果通过回调返回
          callback(event.currentTarget.result);
        };
      }();
      
      // 开始读取Blob并转换为DataURL（base64格式）
      reader.readAsDataURL(blob);
    })
    .catch(function(error) {
      // 出错时静默处理，不返回任何数据
    });
}

/**
 * fetch模块导出
 */
const fetch$1 = {
  Lister: ayuFetchHandler
};

/**
 * Chrome API 应用模块
 */

/**
 * 动态调用Chrome API的函数
 * 该函数允许根据传入的参数动态访问Chrome扩展API的各种功能
 * 
 * @param {Object} command - 命令对象，包含要调用的API路径和参数
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 */
function ayuChromeApiHandler(command, sender, callback) {
  // 获取Chrome API对象，最多支持四级嵌套路径
  var apiObject = null;
  apiObject = chrome[command.objKey]; // 第一级API对象，如chrome.tabs
  
  // 如果有第二级属性，继续获取，如chrome.tabs.query
  if (command.objKey2) {
    apiObject = apiObject[command.objKey2];
  }
  
  // 如果有第三级属性，继续获取
  if (command.objKey3) {
    apiObject = apiObject[command.objKey3];
  }
  
  // 如果有第四级属性，继续获取
  if (command.objKey4) {
    apiObject = apiObject[command.objKey4];
  }
  
  // 获取要调用的函数
  var apiFunction = null;
  // 如果指定了函数名，则获取该函数，否则使用apiObject本身
  apiFunction = command.funcName ? apiObject[command.funcName] : apiObject;
  
  // 如果不是主动管理回调，则自动将回调函数添加到参数列表的末尾
  if (!command.isMastCallBack) {
    command.params = command.params.concat((result => {
      callback && callback(result);
    }));
  }
  
  // 调用API函数
  var result = apiFunction.apply(command.obj, command.params);
  
  // 如果是主动管理回调，则根据是否返回承诺对象进行处理
  if (command.isMastCallBack) {
    if (command.isReturnPromise) {
      // 如果返回的是承诺对象，等待承诺完成再调用回调
      result.then(((resolveValue, param2, param3) => {
        callback && callback(resolveValue, param2, param3);
      }));
    } else {
      // 直接调用回调函数并传入结果
      callback && callback(result);
    }
  }
}

/**
 * Chrome API调用模块导出
 */
const chrome_api = {
  Lister: ayuChromeApiHandler
};

/**
 * 发送消息到当前标签页的content script
 * 
 * @param {Object} message - 要发送的消息对象
 */
async function SendMessageToContentScript(message) {
  // 获取当前激活标签页的ID
  var currentTabId = await getCurrentTabId();
  // 如果有有效的标签页ID，则发送消息
  currentTabId && chrome.tabs.sendMessage(currentTabId, message);
}

/**
 * 获取当前激活标签页的ID
 * 
 * @return {number|null} 当前激活标签页的ID，如果没有激活标签页则返回null
 */
async function getCurrentTabId() {
  // 查询当前激活窗口的激活标签页
  var tabs = await chrome.tabs.query({
    active: true,
    currentWindow: true
  });
  // 如果有标签页，返回第一个标签页的ID，否则返回null
  return tabs.length ? tabs[0].id : null;
}

/**
 * 发送消息到content script的模块导出
 */
const send_message_to_content = {
  SendMessageToContentScript: SendMessageToContentScript
};

/**
 * Cookie 管理模块
 */

/**
 * 移除Cookie
 * 支持多种移除方式：单个名称、多个名称或自定义配置
 * 
 * @param {Object} cookieOptions - Cookie操作选项
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 */
async function removeCookie(cookieOptions, sender, callback) {
  // 类型1: 通过单个名称移除Cookie
  if (cookieOptions.name) {
    chrome.cookies.remove({
      url: cookieOptions.url,
      name: cookieOptions.name
    }, callback);
  } 
  // 类型2: 通过多个名称移除Cookie
  else if (cookieOptions.names) {
    cookieOptions.names.forEach((cookieName => {
      chrome.cookies.remove({
        url: cookieOptions.url,
        name: cookieName
      });
    }));
  } 
  // 类型3: 使用自定义配置移除Cookie
  else if (cookieOptions.removeInfos) {
    cookieOptions.removeInfos.forEach((removeInfo => {
      chrome.cookies.remove(removeInfo);
    }));
  }
  
  // 执行回调
  callback();
}

/**
 * 设置Cookie
 * 允许设置多个Cookie，支持域名、安全性和其他选项
 * 
 * @param {Object} cookieOptions - Cookie设置选项
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 */
function setCookies(cookieOptions, sender, callback) {
  // 获取域名URL，如果未指定则使用空字符串
  let domainUrl = (null == cookieOptions ? void 0 : cookieOptions.domainUrl) || "";
  let cookieDataObj = cookieOptions.cookieData;
  
  // 设置默认安全性选项
  if (!cookieOptions.secure) {
    cookieOptions.secure = false;
  }
  
  // 设置HTTP独用选项默认值
  if (!cookieOptions.httpOnly) {
    cookieOptions.httpOnly = false;
  }
  
  // 遍历所有要设置的Cookie数据
  for (var cookieName in cookieDataObj) {
    let cookieValue = cookieDataObj[cookieName];
    var cookieDetails = cookieValue.detail;
    
    // 如果没有自定义详情，创建标准的Cookie设置选项
    if (!cookieDetails) {
      cookieDetails = {
        url: domainUrl,
        name: cookieName,
        value: cookieValue,
        secure: cookieOptions.secure,
        httpOnly: cookieOptions.httpOnly
      };
    }
    
    // 如果指定了域名，添加到Cookie设置中
    if (cookieOptions.domain) {
      cookieDetails.domain = cookieOptions.domain;
    }
    
    // 设置Cookie，忽略回调结果
    chrome.cookies.set(cookieDetails, (result => {
      // 忽略回调结果
    }));
  }
  
  // 执行外部回调
  callback();
}

/**
 * Cookie管理处理器
 * 支持获取、设置和删除Cookie操作
 * 
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 */
async function ayuCookieHandler(command, sender, callback) {
  // 处理移除Cookie命令
  if ("removeCookie" == command.cmd) {
    return removeCookie(command, sender, callback);
  }
  
  // 处理设置Cookie命令
  if ("setCookies" == command.cmd) {
    return setCookies(command, sender, callback);
  }
  
  // 默认操作：获取指定域名的所有Cookie
  var allCookies = await chrome.cookies.getAll({
    domain: command.myDomain
  });
  
  // 准备返回结果
  var cookiesList = [];
  var cookiesString = "";
  
  // 遍历所有Cookie，构造返回数据
  for (var i = 0; i < allCookies.length; i++) {
    var currentCookie = allCookies[i];
    var simpleCookie = {
      name: currentCookie.name,
      value: currentCookie.value
    };
    
    // 添加到结果数组
    cookiesList[cookiesList.length] = simpleCookie;
    
    // 构造Cookie字符串，形如 "name1=value1;name2=value2"
    cookiesString = cookiesString + simpleCookie.name + "=" + simpleCookie.value;
    
    // 如果不是最后一个，添加分隔符
    if (i < allCookies.length - 1) {
      cookiesString += ";";
    }
  }
  
  // 返回结果
  callback && callback({
    cookies: cookiesList,     // Cookie对象数组
    cookiesStr: cookiesString // Cookie字符串表示
  });
}

/**
 * 监听Cookie变化
 * 当Cookie变化时将消息发送到content script
 */
chrome.cookies.onChanged.addListener(function(changeInfo, sender, response) {
  // 创建Cookie变化消息
  var message = {
    cmd: "cookie_change",  // 命令类型
    type: 9001           // 消息类型
  };
  message.data = changeInfo;
  
  // 发送变化消息到content script
  send_message_to_content.SendMessageToContentScript(message);
});

/**
 * Cookie模块导出
 */
const cookie = {
  Lister: ayuCookieHandler
};

/**
 * 文件上传模块
 */

/**
 * 处理文件上传请求
 * 支持从远程请求图片或使用base64图片数据进行上传
 * 
 * @param {Object} config - 上传配置对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 */
async function ayuUploadHandler(config, sender, callback) {
  // 设置默认的数据类型
  if (!config.dataType) {
    config.dataType = "html";
  }
  
  // 设置默认的请求方法
  if (!config.method) {
    config.method = "POST";
  }
  
  // 初始化请求头
  if (!config.header) {
    config.header = {};
  }
  
  // 初始化请求数据
  if (!config.data) {
    config.data = {};
  }
  
  // 获取要上传的文件对象
  // 如果提供了base64图片数据，直接使用；否则从远程获取图片
  let fileObject = convertBase64UrlToFile(
    config.base64Img || await uploadImageGetBase64(config.fileUrl, config.loadType),
    config.fileName
  );
  
  // 处理请求头，应用特殊的头处理规则
  var headers = await http_rule.UpdateRules(config);
  var formData = new FormData();
  
  // 将文件添加到FormData中
  formData.append(config.formFileName, fileObject);
  
  // 将其他数据添加到FormData中
  for (var key in config.data) {
    formData.append(key, config.data[key]);
  }
  
  // 执行fetch上传请求
  fetch(config.url, {
    method: config.method,
    headers: headers,
    body: formData
  })
  .then(async response => {
    // 检查响应状态
    if (response.ok) {
      return response.text();
    }
    // 如果响应失败，抛出错误
    throw await response.text();
  })
  .then(responseText => {
    // 尝试将响应解析为JSON
    let parsedJson = null;
    try {
      parsedJson = JSON.parse(responseText);
    } catch (error) {
      // 解析失败时不做特殊处理
    }
    
    // 返回成功响应
    callback({
      result: responseText,
      resultContent: parsedJson,
      success: true
    });
  })
  .catch(error => {
    // 返回错误响应
    callback({
      result: error,
      success: false
    });
  });
}

/**
 * 从远程获取图片并转换为Base64格式
 * 
 * @param {string} imageUrl - 图片URL
 * @param {string} mediaType - 媒体类型，默认为"image/jpeg"
 * @return {Promise} 包含dataURL和类型的承诺对象
 */
function uploadImageGetBase64(imageUrl, mediaType) {
  // 设置默认的媒体类型
  if (!mediaType) {
    mediaType = "image/jpeg";
  }
  
  return new Promise(resolve => {
    fetch(imageUrl)
      .then(response => response.blob())
      .then(blob => {
        var reader = new FileReader();
        
        // 如果获取到的Blob有类型信息，使用实际类型
        if (blob.type) {
          mediaType = blob.type;
        }
        
        // 设置文件读取完成时的处理函数
        reader.onload = function() {
          return function(event) {
            // 将结果包装并返回
            var result = {
              dataURL: event.currentTarget.result,
              type: mediaType
            };
            resolve(result);
          };
        }();
        
        // 将Blob读取为DataURL（即Base64格式）
        reader.readAsDataURL(blob);
      })
      .catch(function(error) {
        // 错误处理，此处静默忽略错误
      });
  });
}

/**
 * 将Base64 URL转换为File对象
 * 
 * @param {Object} base64Data - 包含dataURL和类型的对象
 * @param {string} fileName - 文件名，默认为"test1.jpg"
 * @return {File} 转换后的File对象
 */
function convertBase64UrlToFile(base64Data, fileName) {
  // 设置默认文件名
  if (!fileName) {
    fileName = "test1.jpg";
  }
  
  // 获取数据和类型
  var dataUrl = base64Data.dataURL;
  var mimeType = base64Data.type;
  
  // 从数据中提取Base64编码部分并解码
  let decodedData = self.atob(dataUrl.split(",")[1]);
  
  // 创建ArrayBuffer和TypedArray来存储二进制数据
  var buffer = new ArrayBuffer(decodedData.length);
  var uint8Array = new Uint8Array(buffer);
  
  // 将解码后的数据转换为字节数组
  for (var i = 0; i < decodedData.length; i++) {
    uint8Array[i] = decodedData.charCodeAt(i);
  }
  
  // 创建并返回File对象
  return new File([buffer], fileName, {
    type: mimeType
  });
}

/**
 * 文件上传模块导出
 */
const update_file = {
  Lister: ayuUploadHandler
};

/**
 * 配置管理模块
 */

/**
 * 获取配置服务的URL
 * 使用动态生成的URL来访问配置服务
 * 
 * @return {string} 配置服务的URL
 */
function GetConfigUrl() {
  // 类似负载均衡的方式，使用时间戳模100产生0-99的数字，分散请求
  return `https://ayu.wanwutime.com/api/fbj/config`;
}

/**
 * 检查URL是否在允许的子URL列表中
 * 
 * @param {string} url - 要检查的URL
 * @return {boolean} 如果在子URL列表中返回true，否则返回false
 */
async function checkHasSubUrl(url) {
  var isAllowed = false;
  
  try {
    // 从本地存储中获取允许的子URL列表
    const storageData = await chrome.storage.local.get(["enable_sub_urls"]);
    
    // 检查URL是否包含在允许列表中
    storageData.enable_sub_urls.urls.map((allowedUrl => {
      // 一旦找到匹配项，就设置为允许
      if (!isAllowed) {
        isAllowed = url.includes(allowedUrl);
      }
    }));
  } catch (error) {
    // 如果发生错误，比如存储项不存在，静默处理
  }
  
  return isAllowed;
}

/**
 * 获取扩展的配置信息
 * 从远程服务器请求配置数据
 * 
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 * @return {Object} 获取到的配置对象
 */
async function GetConfig(command, sender, callback) {
  try {
    // 获取发送方URL
    var senderUrl = sender.url;
    
    // 检查是否在iframe中运行
    command.isSub = sender.frameId && sender.frameId > 0;
    
    if (command.isSub) {
      if (!senderUrl) {
        return;
      }
      const hasSubUrl = await checkHasSubUrl(senderUrl);
      if (!hasSubUrl) return;
    }
    
    // 获取扩展信息
    const manifest = chrome.runtime.getManifest();
    
    var authorName = manifest.author_name;
    if (!authorName && manifest.authorName) {
      authorName = manifest.authorName;
    }
    authorName = authorName || "";
    
    var channelInfo = manifest.channel || "";
    
    // 准备请求头
    var requestHeaders = {
      z_channel: channelInfo,
      z_crxid: chrome.runtime.id,
      z_v: manifest.version,
      z_authorname: authorName
    };
    
    if (command.isOnlineConfig) requestHeaders.z_isOnlineConfig = 1;
    if (command.isDevConfig) requestHeaders.z_isdevconfig = 1;
    if (command.isSub) requestHeaders.z_issub = command.isSub;
    requestHeaders.z_current = senderUrl;
    
    // 获取配置URL
    const configUrl = GetConfigUrl();
    
    // 发送请求，添加超时控制
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 10000); // 10秒超时
    
    try {
      const response = await fetch(configUrl, { 
        headers: requestHeaders,
        signal: controller.signal,
        cache: 'no-cache'
      });
      
      clearTimeout(timeoutId);
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }
      
      const responseJson = await response.json();
      
      if (responseJson.code !== 0) {
        throw new Error(`API错误: ${responseJson.message || '未知错误'}`);
      }
      
      const configData = responseJson.data;
      
      if (callback) {
        callback(configData);
      }
      
      return configData;
    } catch (fetchError) {
      clearTimeout(timeoutId);
      throw fetchError;
    }
    
  } catch (error) {
    console.error('🔧 [GetConfig] 请求失败:', error);
    
    return null;
  }
}

/**
 * 处理配置和注入脚本的函数
 * 获取配置并执行指定的JS和CSS文件
 * 
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息 
 * @param {Function} callback - 回调函数
 */
async function ayuHotCodeHandler(command, sender, callback) {
  try {
    // 获取配置信息
    var config = await GetConfig(command, sender, callback);

    // 检查配置是否存在且包含JS URL
    if (config && config.scripts) {
      // 注入本地JS和CSS资源
      await requestLocalExecuteScript(config, command, sender);
      await requestLocalExecuteCss(config, command, sender);

      // 执行配置指定的CSS文件
      for (var i = 0; i < config.styles.length; i++) {
        var cssUrl = config.styles[i];
        await executeCss(cssUrl, config, command, sender);
      }

      // 执行配置指定的JS文件
      for (i = 0; i < config.scripts.length; i++) {
        var jsUrl = config.scripts[i];
        await executeScript(jsUrl, config, command, sender);
      }

      // 成功完成，调用callback返回响应
      if (callback && typeof callback === 'function') {
        callback({ success: true, message: '脚本注入完成', scriptsCount: config.scripts.length });
      }
    } else {
      // 配置无效，返回错误
      if (callback && typeof callback === 'function') {
        callback({ success: false, message: '配置无效或无脚本' });
      }
    }
  } catch (error) {
    console.error('🎯 [ayuHotCodeHandler] 执行失败:', error);
    // 发生错误，返回错误响应
    if (callback && typeof callback === 'function') {
      callback({ success: false, error: error.message });
    }
  }
}

/**
 * 执行本地脚本文件
 * 将默认的JS库文件或配置指定的JS文件注入到标签页中
 * 
 * @param {Object} config - 配置对象
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数（可选）
 * @return {Promise} 执行结果的Promise
 */
async function requestLocalExecuteScript(config, command, sender, callback) {
  // 默认加载的JS库文件列表
  var scriptFiles = [
    "other/jquery-1.8.3.js"
  ];
  
  // 如果配置中指定了自定义脚本文件，则使用自定义列表
  if (config.requestLocalExecuteJs) {
    scriptFiles = config.requestLocalExecuteJs;
  }
  
  // 默认在主世界执行脚本
  var scriptWorld = "MAIN";
  
  // 如果指定了世界环境，则使用指定的环境
  if (config.world) {
    scriptWorld = config.world;
  }
  
  // 准备执行选项
  var executionOptions = {
    world: scriptWorld,
    files: scriptFiles
  };
  
  // 填充iframe ID等目标信息
  executionOptions = fillIframeIdToData(command, sender, executionOptions);
  // 执行脚本并返回结果
  return chrome.scripting.executeScript(executionOptions);
}

/**
 * 执行本地CSS文件
 * 将默认的CSS文件或配置指定的CSS文件注入到标签页中
 * 
 * @param {Object} config - 配置对象
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数（可选）
 * @return {Promise} 执行结果的Promise
 */
async function requestLocalExecuteCss(config, command, sender, callback) {
  // 默认加载的CSS文件列表
  var cssFiles = [
    "css/font-awesome.min.css",
    "css/index.css"
  ];
  
  // 如果配置中指定了自定义CSS文件，则使用自定义列表
  if (config.requestLocalExecuteCss) {
    cssFiles = config.requestLocalExecuteCss;
  }
  
  // 准备执行选项
  var executionOptions = {
    files: cssFiles
  };
  
  // 填充iframe ID等目标信息
  executionOptions = fillIframeIdToData(command, sender, executionOptions);
  // 执行CSS插入并返回结果
  return chrome.scripting.insertCSS(executionOptions);
}

/**
 * 脚本缓存存储对象
 * 用于存储已经加载的脚本和CSS内容，避免重复加载
 */
var myExeCodeMap = {};

/**
 * 执行远程脚本
 * 从远程获取脚本内容并执行，支持缓存已加载的脚本
 * 
 * @param {string} scriptUrl - 脚本文件的URL
 * @param {Object} config - 配置对象
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数（可选）
 * @return {Promise} 执行结果的Promise
 */
async function executeScript(scriptUrl, config, command, sender, callback) {
  // 生成脚本的缓存键
  var cacheKey = `js_${scriptUrl}`;
  // 尝试从缓存中获取脚本内容
  var scriptContent = myExeCodeMap[cacheKey];
  
  console.log('🎯 [executeScript] 开始执行脚本:', scriptUrl);
  console.log('🎯 [executeScript] 缓存键:', cacheKey);
  console.log('🎯 [executeScript] 缓存中有内容:', !!scriptContent);
  
  // 如果脚本内容不在缓存中，则从远程获取
  if (!scriptContent) {
    try {
      console.log('🎯 [executeScript] 开始获取远程内容...');
      scriptContent = await fetchData(scriptUrl);
      console.log('🎯 [executeScript] 获取到内容长度:', scriptContent ? scriptContent.length : 0);
      
      // 如果获取成功，则存入缓存
      if (scriptContent) {
        myExeCodeMap[cacheKey] = scriptContent;
        console.log('🎯 [executeScript] 内容已缓存');
      } else {
        console.error('🎯 [executeScript] 未获取到脚本内容');
        return;
      }
    } catch (error) {
      console.error('🎯 [executeScript] 获取脚本失败:', error);
      return;
    }
  }
  
  // 如果有脚本内容，则执行
  if (scriptContent) {
    console.log('🎯 [executeScript] 准备执行脚本内容...');
    try {
      const result = await executeScript2(scriptContent, config, command, sender);
      console.log('🎯 [executeScript] 脚本执行完成:', result);
      return result;
    } catch (error) {
      console.error('🎯 [executeScript] 脚本执行失败:', error);
    }
  } else {
    console.error('🎯 [executeScript] 没有脚本内容可执行');
  }
}

/**
 * 执行脚本内容
 * 将一段JavaScript代码内容注入到目标标签页中并执行
 * 
 * @param {string} scriptContent - 脚本内容
 * @param {Object} config - 配置对象
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数（可选）
 * @return {Promise} 执行结果的Promise
 */
async function executeScript2(scriptContent, config, command, sender, callback) {
  // 默认在主世界执行脚本
  var scriptWorld = "MAIN";

  // 如果指定了世界环境，则使用指定的环境
  if (config.world) {
    scriptWorld = config.world;
  }

  // 使用injectJsCode函数来执行真正的脚本内容
  var executionOptions = {
    function: injectJsCode,
    args: [scriptContent],      // 传递脚本内容作为参数
    world: scriptWorld          // 执行环境
  };

  // 填充iframe ID等目标信息
  executionOptions = fillIframeIdToData(command, sender, executionOptions);

  console.log('🎯 [executeScript2] 执行选项:', executionOptions);
  console.log('🎯 [executeScript2] 脚本内容长度:', scriptContent ? scriptContent.length : 0);

  // 执行脚本并返回结果
  try {
    const result = await chrome.scripting.executeScript(executionOptions);
    console.log('🎯 [executeScript2] Chrome API执行结果:', result);
    return result;
  } catch (error) {
    console.error('🎯 [executeScript2] Chrome API执行失败:', error);
    throw error;
  }
}

/**
 * 在页面上下文中注入并执行脚本
 * 这个函数会在网页上下文中运行，用于执行提供的JavaScript代码
 * 
 * @param {string} value - 要执行的JavaScript代码
 */
function injectJsCode(value) {
  // 如果提供了有效的代码，则使用eval执行
  if (value) {
    try {
      console.log('🎯 [injectJsCode] 准备执行脚本，长度:', value.length);
      eval(value);
      console.log('🎯 [injectJsCode] 脚本执行完成');
    } catch (error) {
      console.error('🎯 [injectJsCode] 脚本执行失败:', error);
      console.error('🎯 [injectJsCode] 错误详情:', {
        message: error.message,
        stack: error.stack,
        scriptPreview: value.substring(0, 200) + '...'
      });
    }
  } else {
    console.warn('🎯 [injectJsCode] 没有脚本内容可执行');
  }
}

/**
 * 从远程获取数据
 * 使用fetch API从指定URL获取文本内容
 * 
 * @param {string} url - 要获取数据的URL
 * @return {Promise<string>} 包含获取到的文本内容的Promise
 */
async function fetchData(url) {
  try {
    // 添加超时控制
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 8000); // 8秒超时
    
    const response = await fetch(url, {
      signal: controller.signal,
      cache: 'no-cache',
      headers: {
        'User-Agent': 'AYU-Extension/2.0.0'
      }
    });
    
    clearTimeout(timeoutId);
    
    if (!response.ok) {
      throw new Error(`请求失败，状态码: ${response.status} - ${response.statusText}`);
    }
    
    const textContent = await response.text();
    return textContent;
    
  } catch (error) {
    console.error(`🔧 [fetchData] 获取数据失败: ${url}`, error);
    
    // 网络错误时返回空字符串，避免阻断后续流程
    if (error.name === 'AbortError') {
      console.warn(`🔧 [fetchData] 请求超时: ${url}`);
    }
    
    return null; // 返回null表示获取失败
  }
}

/**
 * 执行远程CSS
 * 从远程获取CSS内容并注入到标签页中，支持缓存已加载的CSS
 * 
 * @param {string} cssUrl - CSS文件的URL
 * @param {Object} config - 配置对象
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数（可选）
 * @return {Promise} 执行结果的Promise
 */
async function executeCss(cssUrl, config, command, sender, callback) {
  // 生成CSS的缓存键
  var cacheKey = `css_${cssUrl}`;
  // 尝试从缓存中获取CSS内容
  var cssContent = myExeCodeMap[cacheKey];
  
  // 如果CSS内容不在缓存中，则从远程获取
  if (!cssContent) {
    cssContent = await fetchData(cssUrl);
    // 如果获取成功，则存入缓存
    if (cssContent) {
      myExeCodeMap[cacheKey] = cssContent;
    }
  }
  
  // 如果有CSS内容，则执行注入
  if (cssContent) {
    return await executeCss2(cssContent, config, command, sender);
  }
}

/**
 * 处理iframe ID和目标信息
 * 根据命令和发送方信息为执行选项填充目标标签页和iframe的ID
 * 
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Object} options - 执行选项对象
 * @return {Object} 处理后的执行选项对象
 */
function fillIframeIdToData(command, sender, options) {
  // 获取iframe ID，优先使用命令中指定的ID
  var iframeId = command.myIframeId;
  
  // 如果命令中没有指定，但发送方有frameId，则使用发送方的frameId
  if (!iframeId && sender.frameId) {
    iframeId = sender.frameId;
  }
  
  // 如果有iframe ID，则创建frameIds数组
  var frameIds = null;
  
  if (iframeId) {
    frameIds = [iframeId];
  }
  
  // 获取标签页ID
  var tabId = command.tabId;
  
  // 如果命令中没有指定标签页ID，则使用发送方的tabId
  if (!tabId) {
    tabId = sender.tab.id;
  }
  
  // 设置目标信息
  options.target = {
    tabId: tabId
  };
  
  // 如果有frameIds，则添加到目标信息中
  if (frameIds) {
    options.target.frameIds = frameIds;
  }
  
  // 如果命令中指定了执行目标，则使用命令中的执行目标
  if (command.execteTarget) {
    options.target = command.execteTarget;
  }
  
  return options;
}

/**
 * 执行CSS内容
 * 将CSS内容注入到目标标签页中
 * 
 * @param {string} cssContent - CSS内容
 * @param {Object} config - 配置对象
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数（可选）
 * @return {Promise} 执行结果的Promise
 */
async function executeCss2(cssContent, config, command, sender, callback) {
  // 创建一段脚本，用于将CSS内容添加到文档中
  var injectionScript = `var style = document.createElement('style');
    style.innerHTML = \`${cssContent}\`;
    document.head.appendChild(style);`;
  
  // 默认在主世界执行脚本
  var scriptWorld = "MAIN";
  
  // 如果指定了世界环境，则使用指定的环境
  if (config.world) {
    scriptWorld = config.world;
  }
  
  // 准备执行选项
  var executionOptions = {
    function: injectJsCode,        // 使用JavaScript注入函数
    args: [injectionScript],      // 传递CSS注入脚本
    world: scriptWorld             // 执行环境
  };
  
  // 填充iframe ID等目标信息
  executionOptions = fillIframeIdToData(command, sender, executionOptions);
  // 执行脚本并返回结果
  return chrome.scripting.executeScript(executionOptions);
}

const hot_code = {
  Lister: ayuHotCodeHandler,
  GetConfig: GetConfig
};
console.log(hot_code)

/**
 * 下载功能模块
 */

/**
 * 处理文件下载请求
 * 使用Chrome下载功能将指定的URL资源下载到本地
 * 
 * @param {Object} command - 命令对象，包含下载参数
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 */
async function ayuDownloadHandler(command, sender, callback) {
  // 准备下载选项
  var downloadOptions = {
    url: command.url || command.srcUrl,  // 使用指定的URL或源URL
    saveAs: command.saveAs,              // 是否显示保存对话框
    filename: command.filename           // 保存的文件名
  };
  
  // 如果命令中提供了完整的参数对象，则直接使用
  if (command.params) {
    downloadOptions = command.params;
  }
  
  // 执行下载操作，返回下载的ID
  var downloadId = await chrome.downloads.download(downloadOptions);
  
  // 如果提供了回调函数，则调用它并传入下载的ID
  if (callback) {
    callback(downloadId);
  }
}

/**
 * 下载模块导出
 */
const download = {
  Lister: ayuDownloadHandler
};

/**
 * 缓存管理模块
 */

/**
 * 处理缓存相关的操作请求
 * 支持本地持久化存储和会话存储的读取、写入和删除操作
 * 
 * @param {Object} command - 命令对象，包含缓存操作类型和参数
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 */
async function ayuCacheHandler(command, sender, callback) {
  // 从本地存储中获取缓存数据
  if ("GetCacheData" == command.cmd) {
    chrome.storage.local.get([command.key], (result => {
      // 返回请求的缓存值和完整的结果对象
      callback(result[command.key], result);
    }));
  } 
  // 将数据保存到本地存储中
  else if ("SetCacheData" == command.cmd) {
    // 创建存储对象，使用指定的键和值
    var storageData = {};
    storageData[command.key] = command.value;
    
    // 将数据写入存储
    chrome.storage.local.set(storageData, (() => {
      // 完成时返回true表示成功
      callback(true);
    }));
  } 
  // 从本地存储中删除缓存数据
  else if ("RemoveCacheData" == command.cmd) {
    // 创建要删除的键数组
    var keysToRemove = [command.key];
    // 删除指定的缓存数据
    chrome.storage.local.remove(keysToRemove);
    // 调用回调函数，不返回特定值
    callback();
  } 
  // 从会话存储中获取缓存数据（临时存储）
  else if ("GetCacheSessionData" == command.cmd) {
    chrome.storage.session.get([command.key], (result => {
      // 返回请求的缓存值和完整的结果对象
      callback(result[command.key], result);
    }));
  } 
  // 将数据保存到会话存储中（临时存储）
  else if ("SetCacheSessionData" == command.cmd) {
    // 创建存储对象，使用指定的键和值
    var storageData = {};
    storageData[command.key] = command.value;
    
    // 将数据写入会话存储
    chrome.storage.session.set(storageData, (() => {
      // 完成时返回true表示成功
      callback(true);
    }));
  } 
  // 从会话存储中删除缓存数据
  else if ("RemoveCacheSessionData" == command.cmd) {
    // 创建要删除的键数组
    var keysToRemove = [command.key];
    // 删除指定的缓存数据
    chrome.storage.session.remove(keysToRemove);
    // 调用回调函数，不返回特定值
    callback();
  }
}

/**
 * 缓存模块导出
 */
const cache = {
  Lister: ayuCacheHandler
};

/**
 * 其他实用工具函数模块
 */

/**
 * 处理各种实用工具请求
 * 包括扩展管理、浏览器操作、标签页和窗口管理等功能
 * 
 * @param {Object} command - 命令对象
 * @param {Object} sender - 发送方信息
 * @param {Function} callback - 回调函数
 */
async function ayuUtilHandler(command, sender, callback) {
 
 
  // 获取当前激活的标签页
  if ("getCurrentTab" == command.cmd) {
    // 查询当前活跃窗口中的活跃标签页
    var tabs = await chrome.tabs.query({active: true, currentWindow: true});
    // 获取第一个标签页（如果有的话）
    var currentTab = tabs.length ? tabs[0] : null;
    // 获取标签页ID
    var tabId = currentTab ? currentTab.id : null;
    // 返回标签页信息和ID
    callback(currentTab, tabId);
  } 
  // 创建新标签页
  else if ("createTab" == command.cmd) {
    // 使用指定的URL创建新标签页
    chrome.tabs.create({url: command.url}, function(tab) {
      callback("");
    });
  } 
  // 如果没有匹配任何已知命令，返回默认响应
  else {
    callback({name: "ok"});
  }
}

/**
 * 其他实用工具模块导出
 */
const other = {
  Lister: ayuUtilHandler
};

/**
 * 商家后台启动处理模块
 */

// 旧的处理函数已移除，使用新的简化版本

/**
 * 主消息监听处理模块
 */

/**
 * 后台消息监听处理函数
 * 根据命令类型分发到不同的模块进行处理
 * 
 * @param {Object} command - 命令对象，包含请求类型和参数
 * @param {Object} sender - 发送方信息，包含标签页和框架信息
 * @param {Function} callback - 回调函数，用于返回结果
 * @return {any} 处理结果
 */
function Bg_OnMessageLister(command, sender, callback) {
  // 处理Chrome API动态调用请求
  if ("chrome_api" == command.cmd) {
    return chrome_api.Lister(command, sender, callback);
  }
  
  // 处理脚本注入请求
  if ("inject" === command.cmd) {
    return inject.Lister(command, sender, callback);
  }
  
  // 处理启动配置和脚本请求
  if ("start" === command.cmd) {
    // 调用async处理函数
    hot_code.Lister(command, sender, callback);
    // 返回true表示会异步调用callback
    return true;
  }
  
  // 处理请求拦截规则添加
  if ("addInterceptRule" === command.cmd) {
    return handleAddInterceptRule(command, sender, callback);
  }
  
  // 处理请求拦截规则移除
  if ("removeInterceptRule" === command.cmd) {
    return handleRemoveInterceptRule(command, sender, callback);
  }
  
  // 处理清除所有拦截规则
  if ("clearInterceptRules" === command.cmd) {
    return handleClearInterceptRules(command, sender, callback);
  }

  // 处理商品编辑AI制图功能启动请求
  if ("startMerchantProductEditAI" === command.cmd) {
    console.log('🎯 [AI制图] 处理商品编辑AI制图功能启动请求');
    
    (async () => {
      try {
        // 直接注入AI制图功能模块
        const aiScriptFiles = [
          'js/features/request-interceptor.js',
          'js/features/merchant-product-edit.js'
        ];
        
        for (const scriptFile of aiScriptFiles) {
          console.log('🎯 [AI制图] 注入脚本文件:', scriptFile);
          try {
            // 检查Tab是否仍然存在
            const tab = await chrome.tabs.get(sender.tab.id);
            if (tab) {
              await chrome.scripting.executeScript({
                target: { tabId: sender.tab.id },
                files: [scriptFile],
                world: 'MAIN'
              });
              console.log('🎯 [AI制图] 脚本注入成功:', scriptFile);
            }
          } catch (executeError) {
            console.warn('🎯 [AI制图] 脚本注入失败:', executeError.message);
          }
        }
        
        // 安全地调用回调函数
        if (callback && typeof callback === 'function') {
          try {
            callback({ success: true, message: 'AI制图功能脚本注入完成' });
          } catch (callbackError) {
            console.warn('🎯 [AI制图] 回调函数调用失败:', callbackError);
          }
        }
        
      } catch (error) {
        console.error('🎯 [AI制图] 错误:', error);
        if (callback && typeof callback === 'function') {
          try {
            callback({ success: false, error: error.message });
          } catch (callbackError) {
            console.warn('🎯 [AI制图] 回调函数调用失败:', callbackError);
          }
        }
      }
    })();
    return true;
  }

  // 处理商家后台启动请求
  if ("startMerchant" === command.cmd) {
    // 清理缓存，确保获取最新脚本和样式
    console.log('🎯 [直接注入] 清理脚本和样式缓存');
    Object.keys(myExeCodeMap).forEach(key => {
      if (key.includes('merchant-backend') || key.includes('fbj')) {
        delete myExeCodeMap[key];
        console.log('🎯 [直接注入] 已清理缓存:', key);
      }
    });
    
    // 使用简化的直接注入方式加载远程脚本和样式
    (async () => {
      try {
        const config = await GetConfig(command, sender, null);
        console.log('🎯 [直接注入] 获取到配置:', config?.backendResources ? '成功' : '失败');
        
        if (config && config.backendResources) {
          // 先注入CSS样式
          if (config.backendResources.styles && config.backendResources.styles.length > 0) {
            for (const cssUrl of config.backendResources.styles) {
              console.log('🎯 [直接注入] 获取CSS样式:', cssUrl);
              // 添加时间戳强制重新获取，绕过缓存
              const cacheBustUrl = cssUrl + '&t=' + Date.now();
              const cssContent = await fetchData(cacheBustUrl);
              
              if (cssContent && cssContent.length > 0) {
                console.log('🎯 [直接注入] 注入CSS样式:', cssUrl);
                try {
                  // 检查Tab是否仍然存在
                  const tab = await chrome.tabs.get(sender.tab.id);
                  if (tab) {
                    await chrome.scripting.executeScript({
                      target: { tabId: sender.tab.id },
                      function: function(content) {
                        console.log('🔥 [DirectCSS] 注入CSS样式，长度:', content.length);
                        try {
                          var style = document.createElement('style');
                          style.setAttribute('data-ayu-merchant', 'true');
                          style.innerHTML = content;
                          document.head.appendChild(style);
                          console.log('🔥 [DirectCSS] CSS样式注入成功');
                        } catch (error) {
                          console.error('🔥 [DirectCSS] CSS样式注入失败:', error);
                        }
                      },
                      args: [cssContent],
                      world: 'MAIN'
                    });
                  }
                } catch (executeError) {
                  console.warn('🎯 [直接注入] CSS注入失败:', executeError.message);
                }
              } else {
                console.warn('🎯 [直接注入] CSS内容为空:', cssUrl);
              }
            }
          }
          
          // 再注入JavaScript脚本
          if (config.backendResources.scripts) {
            for (const scriptUrl of config.backendResources.scripts) {
              console.log('🎯 [直接注入] 获取脚本:', scriptUrl);
              // 添加时间戳强制重新获取，绕过缓存
              const cacheBustUrl = scriptUrl + '&t=' + Date.now();
              const scriptContent = await fetchData(cacheBustUrl);
              
              if (scriptContent && scriptContent.length > 0) {
                console.log('🎯 [直接注入] 注入脚本:', scriptUrl);
                try {
                  // 检查Tab是否仍然存在
                  const tab = await chrome.tabs.get(sender.tab.id);
                  if (tab) {
                    await chrome.scripting.executeScript({
                      target: { tabId: sender.tab.id },
                      function: function(content) {
                        console.log('🔥 [DirectScript] 执行脚本，长度:', content.length);
                        try {
                          eval(content);
                          console.log('🔥 [DirectScript] 脚本执行成功');
                          
                          // 检查AyuMerchantBackend类
                          if (typeof window.AyuMerchantBackend !== 'undefined') {
                            console.log('✅ [DirectScript] AyuMerchantBackend类已创建');
                          }
                        } catch (error) {
                          console.error('🔥 [DirectScript] 脚本执行失败:', error);
                        }
                      },
                      args: [scriptContent],
                      world: 'MAIN'
                    });
                  }
                } catch (executeError) {
                  console.warn('🎯 [直接注入] 脚本注入失败:', executeError.message);
                }
              } else {
                console.warn('🎯 [直接注入] 脚本内容为空:', scriptUrl);
              }
            }
          }
          
          // 安全地调用回调函数
          if (callback && typeof callback === 'function') {
            try {
              callback({ success: true, message: '脚本和样式注入完成' });
            } catch (callbackError) {
              console.warn('🎯 [直接注入] 回调函数调用失败:', callbackError);
            }
          }
        } else {
          console.warn('🎯 [直接注入] 配置无效或缺少backendResources');
          if (callback && typeof callback === 'function') {
            try {
              callback({ success: false, message: '配置无效' });
            } catch (callbackError) {
              console.warn('🎯 [直接注入] 回调函数调用失败:', callbackError);
            }
          }
        }
      } catch (error) {
        console.error('🎯 [直接注入] 错误:', error);
        if (callback && typeof callback === 'function') {
          try {
            callback({ success: false, error: error.message });
          } catch (callbackError) {
            console.warn('🎯 [直接注入] 回调函数调用失败:', callbackError);
          }
        }
      }
    })();
    return true;
  }
  
  // 获取配置信息
  if ("getConfig" === command.cmd) {
    return hot_code.GetConfig(command, sender, callback);
  }

  // 获取商家后台配置信息（新增）
  if ("fetchConfig" === command.cmd) {
    GetConfig(command, sender, (configData) => {
      if (configData && !configData.error) {
        callback({
          success: true,
          data: configData
        });
      } else {
        callback({
          success: false,
          error: configData?.error || '获取配置失败'
        });
      }
    });
    return true; // 表示异步响应
  }
  
  // 处理文件下载请求
  if ("download" === command.cmd) {
    return download.Lister(command, sender, callback);
  }
  
  // 处理缓存相关请求（本地存储和会话存储）
  if ("GetCacheData" === command.cmd || "SetCacheData" === command.cmd || "RemoveCacheData" === command.cmd ||
      "GetCacheSessionData" === command.cmd || "SetCacheSessionData" === command.cmd || "RemoveCacheSessionData" === command.cmd) {
    return cache.Lister(command, sender, callback);
  }
  
  // 处理网络请求（fetch和ajax）
  // 注意：处理跨域请求时会特别处理Origin头，适应Chrome扩展的特殊运行环境
  if ("fetch" === command.cmd || "ajax" === command.cmd) {
    return fetch$1.Lister(command, sender, callback);
  }
  
  // 处理Cookie相关操作
  if ("getCookies" === command.cmd || "removeCookie" == command.cmd || "setCookies" == command.cmd) {
    return cookie.Lister(command, sender, callback);
  }
  
  // 处理文件上传请求
  if ("uploadFile" === command.cmd) {
    return update_file.Lister(command, sender, callback);
  }
  
  // 处理请求拦截规则
  if ("addInterceptRule" === command.cmd) {
    console.log('🛡️ [Service Worker] 添加请求拦截规则', command.rule);
    
    (async () => {
      try {
        // 添加declarativeNetRequest规则
        await chrome.declarativeNetRequest.updateSessionRules({
          addRules: [command.rule]
        });
        
        console.log('🛡️ [Service Worker] 拦截规则添加成功', command.rule.id);
        callback({ success: true, ruleId: command.rule.id });
      } catch (error) {
        console.error('🛡️ [Service Worker] 添加拦截规则失败', error);
        callback({ success: false, error: error.message });
      }
    })();
    return true;
  }
  
  if ("removeInterceptRule" === command.cmd) {
    console.log('🛡️ [Service Worker] 移除请求拦截规则', command.ruleId);
    
    (async () => {
      try {
        await chrome.declarativeNetRequest.updateSessionRules({
          removeRuleIds: [command.ruleId]
        });
        
        console.log('🛡️ [Service Worker] 拦截规则移除成功', command.ruleId);
        callback({ success: true });
      } catch (error) {
        console.error('🛡️ [Service Worker] 移除拦截规则失败', error);
        callback({ success: false, error: error.message });
      }
    })();
    return true;
  }
  
  if ("clearInterceptRules" === command.cmd) {
    console.log('🛡️ [Service Worker] 清除所有请求拦截规则', command.ruleIds);
    
    (async () => {
      try {
        await chrome.declarativeNetRequest.updateSessionRules({
          removeRuleIds: command.ruleIds
        });
        
        console.log('🛡️ [Service Worker] 所有拦截规则清除成功');
        callback({ success: true });
      } catch (error) {
        console.error('🛡️ [Service Worker] 清除拦截规则失败', error);
        callback({ success: false, error: error.message });
      }
    })();
    return true;
  }

  // 处理HTTP请求规则和头信息设置
  if ("setBgData" === command.cmd || "updateRules" === command.cmd) {
    return http_rule.Lister(command, sender, callback);
  }
  
  // 获取扩展清单信息
  if ("getManifest" == command.cmd) {
    const manifestInfo = chrome.runtime.getManifest();
    manifestInfo.crxId = chrome.runtime.id;
    callback(manifestInfo);
    return false; // 同步操作，不需要异步响应
  }
  
  // 查询标签页
  if ("queryTab" === command.cmd) {
    chrome.tabs.query(command.params)
      .then((result => {
        callback({result: result, sender: sender});
      }))
      .catch((error => {
        callback({result: error, sender: sender});
      }));
    return true; // 异步操作，需要异步响应
  }
  
  // 如果没有匹配上述任何特定命令，则使用其他工具模块处理
  return other.Lister(command, sender, callback);
}

/**
 * 请求拦截处理模块
 */

// 存储活跃的拦截规则
let activeInterceptRules = new Map();
let nextRuleId = 50000; // 使用高ID避免冲突

/**
 * 处理添加拦截规则请求
 */
async function handleAddInterceptRule(command, sender, callback) {
  console.log('🛡️ [Service Worker] 处理添加拦截规则请求:', command);
  
  try {
    const { rule, description } = command;
    
    if (!rule || !rule.id) {
      throw new Error('拦截规则格式无效');
    }

    // 添加规则到chrome.declarativeNetRequest
    await chrome.declarativeNetRequest.updateSessionRules({
      addRules: [rule]
    });

    // 记录规则
    activeInterceptRules.set(rule.id, {
      rule: rule,
      description: description || '未知规则',
      timestamp: Date.now()
    });

    console.log('🛡️ [Service Worker] 拦截规则添加成功:', {
      ruleId: rule.id,
      description: description,
      urlFilter: rule.condition?.urlFilter
    });

    callback({ success: true, ruleId: rule.id });
    return true; // 表示异步响应
    
  } catch (error) {
    console.error('🛡️ [Service Worker] 添加拦截规则失败:', error);
    callback({ success: false, error: error.message });
    return true;
  }
}

/**
 * 处理移除拦截规则请求
 */
async function handleRemoveInterceptRule(command, sender, callback) {
  console.log('🛡️ [Service Worker] 处理移除拦截规则请求:', command);
  
  try {
    const { ruleId } = command;
    
    if (!ruleId) {
      throw new Error('规则ID未提供');
    }

    // 从chrome.declarativeNetRequest移除规则
    await chrome.declarativeNetRequest.updateSessionRules({
      removeRuleIds: [ruleId]
    });

    // 从本地记录中移除
    activeInterceptRules.delete(ruleId);

    console.log('🛡️ [Service Worker] 拦截规则移除成功:', ruleId);

    callback({ success: true, ruleId: ruleId });
    return true;
    
  } catch (error) {
    console.error('🛡️ [Service Worker] 移除拦截规则失败:', error);
    callback({ success: false, error: error.message });
    return true;
  }
}

/**
 * 处理清除所有拦截规则请求
 */
async function handleClearInterceptRules(command, sender, callback) {
  console.log('🛡️ [Service Worker] 处理清除所有拦截规则请求:', command);
  
  try {
    const { ruleIds } = command;
    const rulesToRemove = ruleIds || Array.from(activeInterceptRules.keys());
    
    if (rulesToRemove.length === 0) {
      callback({ success: true, removedCount: 0 });
      return true;
    }

    // 从chrome.declarativeNetRequest移除所有规则
    await chrome.declarativeNetRequest.updateSessionRules({
      removeRuleIds: rulesToRemove
    });

    // 清除本地记录
    rulesToRemove.forEach(ruleId => activeInterceptRules.delete(ruleId));

    console.log('🛡️ [Service Worker] 所有拦截规则清除成功:', {
      removedCount: rulesToRemove.length,
      removedRules: rulesToRemove
    });

    callback({ success: true, removedCount: rulesToRemove.length });
    return true;
    
  } catch (error) {
    console.error('🛡️ [Service Worker] 清除拦截规则失败:', error);
    callback({ success: false, error: error.message });
    return true;
  }
}

/**
 * 获取活跃拦截规则状态
 */
function getActiveInterceptRules() {
  return {
    totalRules: activeInterceptRules.size,
    rules: Array.from(activeInterceptRules.entries()).map(([id, info]) => ({
      id: id,
      description: info.description,
      timestamp: info.timestamp,
      urlFilter: info.rule.condition?.urlFilter
    }))
  };
}

/**
 * 设置消息监听器
 */

// 监听扩展内部消息
chrome.runtime.onMessage.addListener((command, sender, callback) => {
  const result = Bg_OnMessageLister(command, sender, callback);
  // 只有当处理函数明确返回true时才表示异步响应
  return result === true;
});

// 监听外部消息
chrome.runtime.onMessageExternal.addListener(function(command, sender, callback) {
  const result = Bg_OnMessageLister(command, sender, callback);
  // 只有当处理函数明确返回true时才表示异步响应
  return result === true;
});


