/**
 * AYU引擎 - 商品编辑AI制图功能模块
 * 专为拼多多商家后台商品编辑页面提供AI制图工具
 */

class AyuMerchantProductEdit {
    constructor() {
        this.isInitialized = false;
        this.drawerOriginalState = null;
        this.urlParams = new URLSearchParams(window.location.search);
        this.goodsId = this.urlParams.get('goods_id');
        this.commitId = this.urlParams.get('id');
        
        this.init();
    }

    /**
     * 安全的日志输出方法
     */
    log(level, message, data = null) {
        if (typeof window.AyuLogger !== 'undefined' && window.AyuLogger[level]) {
            window.AyuLogger[level]('MerchantProductEdit', message, data);
        } else {
            const prefix = '🎯 [AYU] [MerchantProductEdit]';
            const logMessage = `${prefix} ${message}`;
            data ? console.log(logMessage, data) : console.log(logMessage);
        }
    }

    /**
     * 初始化AI制图功能
     */
    async init() {
        try {
            console.log('🎨 [AYU AI制图] 开始初始化', {
                goodsId: this.goodsId,
                commitId: this.commitId,
                url: window.location.href
            });
            
            this.log('info', 'AI制图功能开始初始化', {
                goodsId: this.goodsId,
                commitId: this.commitId,
                url: window.location.href
            });

            // 验证必要参数
            if (!this.goodsId || !this.commitId) {
                console.error('🎨 [AYU AI制图] 缺少必要的URL参数，无法初始化', {
                    goodsId: this.goodsId,
                    commitId: this.commitId
                });
                this.log('error', '缺少必要的URL参数，无法初始化', {
                    goodsId: this.goodsId,
                    commitId: this.commitId
                });
                return;
            }

            // 等待页面加载完成
            await this.waitForPageReady();

            // 查找现有的Drawer结构
            await this.findAndPrepareDrawer();

            // 创建悬浮按钮
            console.log('🎨 [AYU AI制图] 开始创建悬浮按钮');
            this.createFloatingButton();

            // 绑定事件监听器
            this.bindEventListeners();

            this.isInitialized = true;
            console.log('🎨 [AYU AI制图] 初始化完成');
            this.log('success', 'AI制图功能初始化完成');

            // 检查URL参数，自动触发AI制图功能
            this.checkAutoTrigger();

        } catch (error) {
            this.log('error', 'AI制图功能初始化失败', error);
        }
    }

    /**
     * 等待页面就绪
     */
    async waitForPageReady() {
        return new Promise((resolve) => {
            if (document.readyState === 'complete') {
                resolve();
                return;
            }

            const checkReady = () => {
                if (document.readyState === 'complete') {
                    resolve();
                } else {
                    setTimeout(checkReady, 100);
                }
            };

            checkReady();
        });
    }

    /**
     * 查找并准备Drawer结构
     */
    async findAndPrepareDrawer() {
        // 等待Drawer元素出现
        let attempts = 0;
        const maxAttempts = 10;

        while (attempts < maxAttempts) {
            const drawer = document.querySelector('div[data-testid="beast-core-drawer"]');
            if (drawer) {
                this.log('success', '找到现有Drawer结构', drawer);
                this.saveDrawerOriginalState(drawer);
                return;
            }

            attempts++;
            await new Promise(resolve => setTimeout(resolve, 500));
        }

        this.log('warn', '未找到现有Drawer结构，将在需要时创建');
    }

    /**
     * 保存Drawer原始状态
     */
    saveDrawerOriginalState(drawer) {
        const outerWrapper = drawer;
        const mask = drawer.querySelector('.Drawer_mask_5-161-0');
        const content = drawer.querySelector('.Drawer_content_5-161-0');

        this.drawerOriginalState = {
            outerWrapper: {
                element: outerWrapper,
                className: outerWrapper.className
                // 不保存innerHTML，避免与React冲突
            },
            mask: mask ? {
                element: mask,
                className: mask.className
            } : null,
            content: content ? {
                element: content,
                className: content.className
                // 不保存innerHTML，让React管理内容
            } : null
        };

        this.log('debug', '已保存Drawer原始状态', {
            hasOuterWrapper: !!this.drawerOriginalState.outerWrapper,
            hasMask: !!this.drawerOriginalState.mask,
            hasContent: !!this.drawerOriginalState.content
        });
    }

    /**
     * 检查是否自动触发AI制图功能
     */
    checkAutoTrigger() {
        const ayuAiImg = this.urlParams.get('ayu_ai_img');
        
        console.log('🎨 [AYU AI制图] URL参数检查:', {
            hasAyuAiImg: !!ayuAiImg,
            ayuAiImgValue: ayuAiImg,
            fullUrl: window.location.href
        });

        if (ayuAiImg === '1') {
            console.log('🎨 [AYU AI制图] 检测到ayu_ai_img=1参数，自动触发AI制图功能');
            this.log('info', '检测到自动触发参数，准备显示AI制图界面');
            
            // 延迟执行以确保页面完全加载和按钮创建完成
            setTimeout(() => {
                this.showAIDrawer();
            }, 500);
        }
    }

    /**
     * 创建悬浮按钮
     */
    createFloatingButton() {
        console.log('🎨 [AYU AI制图] createFloatingButton 被调用');
        
        // 检查是否已存在按钮
        if (document.getElementById('ayu-ai-drawing-button')) {
            console.log('🎨 [AYU AI制图] 按钮已存在，跳过创建');
            this.log('info', 'AI制图悬浮按钮已存在');
            return;
        }

        const button = document.createElement('div');
        button.id = 'ayu-ai-drawing-button';
        button.className = 'ayu-ai-drawing-floating-button';
        button.innerHTML = `
            <div class="ayu-ai-button-content">
                <span class="ayu-ai-button-icon">
                    <svg width="16" height="16" style="vertical-align: text-bottom;" viewBox="0 0 1024 1024">
                        <path d="M916.693333 146.218667a109.226667 109.226667 0 0 0-76.096 30.592 103.296 103.296 0 0 0-31.466666 73.877333v522.56c0 27.797333 11.349333 54.357333 31.466666 73.898667a109.269333 109.269333 0 0 0 76.074667 30.656 109.226667 109.226667 0 0 0 76.096-30.592 103.296 103.296 0 0 0 31.445333-73.877334V250.666667a102.912 102.912 0 0 0-31.445333-73.877334 109.162667 109.162667 0 0 0-76.096-30.592z m-368.576 104.533333A168.256 168.256 0 0 0 485.056 174.933333a175.573333 175.573333 0 0 0-192.192-0.149333 168.256 168.256 0 0 0-63.274667 75.946667L4.416 792.170667a61.098667 61.098667 0 0 0 6.805333 58.090666c5.845333 8.426667 13.76 15.296 22.976 20.053334 9.28 4.821333 19.52 7.317333 29.994667 7.381333h353.621333a66.005333 66.005333 0 0 0 29.866667-7.530667 63.936 63.936 0 0 0 22.890667-20.117333 61.589333 61.589333 0 0 0 6.570666-57.813333l-35.242666-84.693334a63.146667 63.146667 0 0 0-23.573334-28.096 65.834667 65.834667 0 0 0-35.690666-10.752 65.984 65.984 0 0 1-29.866667-7.530666 63.936 63.936 0 0 1-22.890667-20.117334 61.610667 61.610667 0 0 1-6.570666-57.813333c5.333333-12.8 14.421333-23.701333 26.026666-31.296a72.341333 72.341333 0 0 1 79.317334 0c11.712 7.594667 20.778667 18.496 26.026666 31.232L561.066667 839.04c4.821333 11.562667 13.013333 21.397333 23.466666 28.245333a65.109333 65.109333 0 0 0 35.797334 10.602667h93.994666c10.453333-0.064 20.714667-2.581333 29.994667-7.381333 9.173333-4.757333 17.066667-11.669333 22.976-20.138667a61.589333 61.589333 0 0 0 6.656-58.005333L548.117333 250.837333v-0.064z" fill="white"></path>
                    </svg>
                </span>
                <span class="ayu-ai-button-text">阿欲智能制图</span>
            </div>
        `;

        // 添加样式
        button.style.cssText = `
            position: fixed;
            right: 30px;
            top: 30%;
            transform: translateY(-50%);
            z-index: ${window.AyuConfig?.ui?.zIndex?.toolbar || 10000};
            background: #407cff;
            color: white;
            padding: 12px 16px;
            border-radius: 25px;
            cursor: pointer;
            box-shadow: 0 4px 12px rgba(255, 102, 0, 0.3);
            font-size: 14px;
            font-weight: 500;
            user-select: none;
            transition: all 0.3s ease;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        `;

        // 添加悬停效果
        button.addEventListener('mouseenter', () => {
            button.style.transform = 'translateY(-50%) scale(1.05)';
            button.style.boxShadow = '0 6px 20px rgba(255, 102, 0, 0.4)';
        });

        button.addEventListener('mouseleave', () => {
            button.style.transform = 'translateY(-50%) scale(1)';
            button.style.boxShadow = '0 4px 12px rgba(255, 102, 0, 0.3)';
        });

        // 添加点击事件
        button.addEventListener('click', () => {
            this.showAIDrawer();
        });

        document.body.appendChild(button);
        console.log('🎨 [AYU AI制图] 悬浮按钮已添加到页面', button);
        this.log('success', '创建AI制图悬浮按钮完成');
    }

    /**
     * 显示AI制图Drawer
     */
    showAIDrawer() {
        this.log('info', '显示AI制图界面');
        console.log('🎨 [AYU AI制图] 开始显示独立Drawer');

        // 检查是否已经有独立的Drawer
        let existingDrawer = document.getElementById('ayu-ai-drawer');
        if (existingDrawer) {
            existingDrawer.style.display = 'block';
            console.log('🎨 [AYU AI制图] 显示现有Drawer');
            return;
        }

        // 创建完全独立的Drawer结构，不依赖React的DOM
        this.createIndependentDrawer();
    }

    /**
     * 创建独立的Drawer，避免React冲突
     */
    createIndependentDrawer() {
        console.log('🎨 [AYU AI制图] 创建独立Drawer');

        // 生成iframe链接
        const logParams = JSON.stringify({
            //goods_id: this.goodsId,
            //goods_commit_id: this.commitId
        });

        const iframeSrc = `https://aid.pinduoduo.com/design/ai-generator?layout=drawer&source=mms&left_images_total=2&sn_source=Product_Edit&log_params=${encodeURIComponent(logParams)}&currentTab=AiGenerator&subTab=AiSellPoints,AiGenerator,ClarityEnhance,Image2Video`;

        // 创建独立的Drawer HTML结构
        const drawerHTML = `
            <div class="ayu-ai-drawer-mask" style="
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0, 0, 0, 0.5);
                z-index: 999998;
            "></div>
            <div class="ayu-ai-drawer-content" style="
                position: fixed;
                top: 0;
                right: 0;
                width: 1000px;
                height: 100%;
                background: white;
                z-index: 999999;
                box-shadow: -2px 0 8px rgba(0, 0, 0, 0.15);
                transform: translateX(0);
                transition: transform 0.3s ease;
            ">
                <div style="width: 100%; height: 100%; display: flex; flex-direction: column;">
                    <div style="height: 48px; align-items: center; display: flex; justify-content: space-between; padding: 0px 16px; border-bottom: 1px solid #f0f0f0; flex-shrink: 0;">
                        <div style="font-size: 16px; color: rgba(0, 0, 0, 0.8); line-height: 16px; font-weight: 400;">
                            阿欲AI制图
                        </div>
                        <svg class="ayu-ai-close-btn" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1024 1024" 
                             style="cursor: pointer; margin-right: 8px; width: 16px; height: 16px; fill: #666;">
                            <path d="M566.306 512l244.376-244.376c14.997-14.996 14.997-39.309 0-54.305-14.996-14.997-39.309-14.997-54.305 0L512 457.694 267.624 213.318c-14.996-14.997-39.31-14.997-54.306 0-14.996 14.996-14.996 39.309 0 54.305L457.694 512 213.318 756.376c-14.996 14.996-14.996 39.31 0 54.306 14.996 14.996 39.31 14.996 54.306 0L512 566.306l244.376 244.376c14.996 14.996 39.309 14.996 54.305 0 14.997-14.996 14.997-39.31 0-54.306L566.306 512z"></path>
                        </svg>
                    </div>
                    <iframe src="${iframeSrc}" 
                            width="100%" 
                            height="100%" 
                            style="border: none; display: block; flex: 1;">
                    </iframe>
                </div>
            </div>
        `;

        // 创建独立的Drawer容器
        const drawerContainer = document.createElement('div');
        drawerContainer.id = 'ayu-ai-drawer';
        drawerContainer.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            z-index: 999997;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        `;
        drawerContainer.innerHTML = drawerHTML;

        // 添加到页面
        document.body.appendChild(drawerContainer);

        // 绑定事件
        this.bindIndependentDrawerEvents(drawerContainer);

        // 启用请求拦截
        this.enableRequestInterception();


        console.log('🎨 [AYU AI制图] 独立Drawer创建完成');
        this.log('success', 'AI制图界面已显示');
    }


    /**
     * 绑定独立Drawer的事件
     */
    bindIndependentDrawerEvents(drawerContainer) {
        console.log('🎨 [AYU AI制图] 绑定独立Drawer事件');

        // 关闭按钮点击事件
        const closeBtn = drawerContainer.querySelector('.ayu-ai-close-btn');
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                console.log('🎨 [AYU AI制图] 点击关闭按钮');
                this.hideIndependentDrawer();
            });
        }

        // 遮罩层点击事件
        const mask = drawerContainer.querySelector('.ayu-ai-drawer-mask');
        if (mask) {
            mask.addEventListener('click', (e) => {
                if (e.target === mask) {
                    console.log('🎨 [AYU AI制图] 点击遮罩层关闭');
                    this.hideIndependentDrawer();
                }
            });
        }

        // ESC键事件
        const handleEscKey = (e) => {
            if (e.key === 'Escape') {
                console.log('🎨 [AYU AI制图] 按ESC键关闭');
                this.hideIndependentDrawer();
                document.removeEventListener('keydown', handleEscKey);
            }
        };
        document.addEventListener('keydown', handleEscKey);

        // 保存ESC事件处理器的引用，方便后续清理
        this.escKeyHandler = handleEscKey;

        this.log('debug', '独立Drawer关闭事件已绑定');
    }

    /**
     * 隐藏独立的Drawer
     */
    hideIndependentDrawer() {
        console.log('🎨 [AYU AI制图] 隐藏独立Drawer');

        const drawerContainer = document.getElementById('ayu-ai-drawer');
        if (drawerContainer) {
            drawerContainer.style.display = 'none';
        }

        // 清理ESC键监听器
        if (this.escKeyHandler) {
            document.removeEventListener('keydown', this.escKeyHandler);
            this.escKeyHandler = null;
        }

        // 禁用请求拦截
        this.disableRequestInterception();

        this.log('success', 'AI制图界面已隐藏');
    }

    /**
     * 绑定Drawer关闭事件 (原方法保留兼容性)
     */
    bindDrawerCloseEvents() {
        // 关闭按钮点击事件
        const closeBtn = document.querySelector('.ayu-drawer-close-btn');
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                this.hideAIDrawer();
            });
        }

        // 遮罩层点击事件
        const mask = document.querySelector('.Drawer_mask_5-161-0');
        if (mask) {
            mask.addEventListener('click', (e) => {
                if (e.target === mask) {
                    this.hideAIDrawer();
                }
            });
        }

        // ESC键事件
        const handleEscKey = (e) => {
            if (e.key === 'Escape') {
                this.hideAIDrawer();
                document.removeEventListener('keydown', handleEscKey);
            }
        };
        document.addEventListener('keydown', handleEscKey);

        this.log('debug', 'Drawer关闭事件已绑定');
    }

    /**
     * 隐藏AI制图Drawer
     */
    hideAIDrawer() {
        this.log('info', '隐藏AI制图界面');

        if (!this.drawerOriginalState) {
            this.log('error', '未找到Drawer原始状态');
            return;
        }

        // 使用setTimeout确保在React渲染周期外执行
        setTimeout(() => {
            try {
                const drawer = document.querySelector('div[data-testid="beast-core-drawer"]');
                if (!drawer) {
                    this.log('error', '未找到Drawer容器');
                    return;
                }

                // 恢复原始状态
                const outerWrapper = this.drawerOriginalState.outerWrapper.element;
                const mask = this.drawerOriginalState.mask?.element;
                const content = this.drawerOriginalState.content?.element;

                // 使用requestAnimationFrame确保DOM操作的时机
                requestAnimationFrame(() => {
                    try {
                        // 移除可见性类
                        if (outerWrapper) {
                            outerWrapper.className = this.drawerOriginalState.outerWrapper.className;
                        }
                        if (mask) {
                            mask.className = this.drawerOriginalState.mask.className;
                        }
                        if (content) {
                            content.className = this.drawerOriginalState.content.className;
                            // 使用更现代的方式清空内容，避免React冲突
                            content.replaceChildren();
                        }

                        this.log('success', 'AI制图界面已隐藏，状态已恢复');
                    } catch (innerError) {
                        this.log('error', 'DOM恢复操作失败', innerError);
                    }
                });

            } catch (error) {
                this.log('error', '隐藏AI制图界面时出错', error);
            }
        }, 50);
    }

    /**
     * 绑定全局事件监听器
     */
    bindEventListeners() {
        // 页面卸载时的清理
        window.addEventListener('beforeunload', () => {
            this.cleanup();
        });

        this.log('debug', '全局事件监听器已绑定');
    }

    /**
     * 启用请求拦截
     */
    async enableRequestInterception() {
        console.log('🛡️ [AYU AI制图] 启用请求拦截');
        
        this.log('info', '网络拦截由Service Worker的declarativeNetRequest处理');
        this.log('success', '请求拦截配置完成');
    }


    /**
     * 禁用请求拦截
     */
    async disableRequestInterception() {
        console.log('🛡️ [AYU AI制图] 禁用请求拦截');
        
        this.log('info', '网络拦截由Service Worker管理，无需手动禁用');
    }

    /**
     * 清理资源
     */
    cleanup() {
        // 移除悬浮按钮
        const button = document.getElementById('ayu-ai-drawing-button');
        if (button) {
            button.remove();
        }

        // 移除独立的Drawer
        const drawerContainer = document.getElementById('ayu-ai-drawer');
        if (drawerContainer) {
            drawerContainer.remove();
        }

        // 清理ESC键监听器
        if (this.escKeyHandler) {
            document.removeEventListener('keydown', this.escKeyHandler);
            this.escKeyHandler = null;
        }

        // 恢复原始Drawer状态（如果有的话）
        if (this.drawerOriginalState) {
            this.hideAIDrawer();
        }

        this.log('debug', '资源清理完成');
    }

    /**
     * 获取功能状态
     */
    getStatus() {
        return {
            isInitialized: this.isInitialized,
            goodsId: this.goodsId,
            commitId: this.commitId,
            hasDrawerState: !!this.drawerOriginalState
        };
    }
}

// 创建全局实例
window.AyuMerchantProductEdit = new AyuMerchantProductEdit();

// 触发加载完成事件
document.dispatchEvent(new CustomEvent('ayuMerchantProductEditLoaded', {
    detail: { loaded: true }
}));