/**
 * AYU引擎 - AI制图页面优化模块
 * 专为https://aid.pinduoduo.com/design/ai-generator页面定制
 */

class AyuAIGeneratorPage {
    constructor() {
        this.isInitialized = false;
        this.observer = null;
        
        this.init();
    }

    /**
     * 安全的日志输出方法
     */
    log(level, message, data = null) {
        if (typeof window.AyuLogger !== 'undefined' && window.AyuLogger[level]) {
            window.AyuLogger[level]('AIGeneratorPage', message, data);
        } else {
            const prefix = '🎨 [AYU] [AIGeneratorPage]';
            const logMessage = `${prefix} ${message}`;
            data ? console.log(logMessage, data) : console.log(logMessage);
        }
    }

    /**
     * 初始化AI制图页面优化
     */
    async init() {
        try {
            this.log('info', 'AI制图页面优化开始初始化', {
                url: window.location.href,
                title: document.title
            });

            // 检查是否为AI制图页面
            if (!this.isAIGeneratorPage()) {
                this.log('warn', '不是AI制图页面，跳过初始化');
                return;
            }

            // 等待页面加载完成
            await this.waitForPageReady();

            // 隐藏使用教程按钮
            this.hideTutorialButton();

            // 设置监听器监听动态内容变化
            this.setupContentObserver();

            this.isInitialized = true;
            this.log('success', 'AI制图页面优化初始化完成');

        } catch (error) {
            this.log('error', 'AI制图页面优化初始化失败', error);
        }
    }

    /**
     * 检查是否为AI制图页面
     */
    isAIGeneratorPage() {
        const url = window.location.href;
        return url.includes('aid.pinduoduo.com/design/ai-generator');
    }

    /**
     * 等待页面就绪
     */
    async waitForPageReady() {
        return new Promise((resolve) => {
            if (document.readyState === 'complete') {
                resolve();
                return;
            }

            const checkReady = () => {
                if (document.readyState === 'complete') {
                    resolve();
                } else {
                    setTimeout(checkReady, 100);
                }
            };

            checkReady();
        });
    }

    /**
     * 隐藏使用教程按钮
     */
    hideTutorialButton() {
        const hideButton = () => {
            // 查找使用教程按钮容器
            const tutorialContainer = document.querySelector('.aig__tips-course');
            
            if (tutorialContainer) {
                tutorialContainer.style.display = 'none';
                this.log('success', '已隐藏使用教程按钮');
                return true;
            }

            // 也可以通过更精确的选择器查找
            const tutorialButton = document.querySelector('a.BTN_outerWrapper_5-125-0.BTN_textPrimary_5-125-0.BTN_small_5-125-0.BTN_outerWrapperLink_5-125-0');
            if (tutorialButton && tutorialButton.textContent.includes('使用教程')) {
                // 隐藏整个父容器
                const parent = tutorialButton.closest('.aig__tips-course') || tutorialButton.parentElement;
                if (parent) {
                    parent.style.display = 'none';
                    this.log('success', '通过父元素已隐藏使用教程按钮');
                    return true;
                } else {
                    tutorialButton.style.display = 'none';
                    this.log('success', '直接隐藏使用教程按钮');
                    return true;
                }
            }

            return false;
        };

        // 立即尝试隐藏
        if (!hideButton()) {
            // 如果没找到，等待一段时间后再试
            setTimeout(() => {
                if (!hideButton()) {
                    this.log('warn', '未找到使用教程按钮，可能页面结构已变化');
                }
            }, 1000);
        }
    }

    /**
     * 设置内容监听器，监听动态加载的内容
     */
    setupContentObserver() {
        // 创建 MutationObserver 监听DOM变化
        this.observer = new MutationObserver((mutations) => {
            let shouldCheckButton = false;
            
            mutations.forEach((mutation) => {
                // 检查是否有新节点添加
                if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                    mutation.addedNodes.forEach((node) => {
                        // 检查新添加的节点是否包含我们要隐藏的元素
                        if (node.nodeType === Node.ELEMENT_NODE) {
                            const hasTargetClass = node.classList?.contains('aig__tips-course') ||
                                                 node.querySelector?.('.aig__tips-course');
                            
                            if (hasTargetClass) {
                                shouldCheckButton = true;
                            }
                        }
                    });
                }
            });

            if (shouldCheckButton) {
                this.log('debug', '检测到相关DOM变化，重新检查教程按钮');
                setTimeout(() => this.hideTutorialButton(), 100);
            }
        });

        // 开始监听整个文档的变化
        this.observer.observe(document.body, {
            childList: true,
            subtree: true
        });

        this.log('debug', '内容监听器已设置');
    }

    /**
     * 清理资源
     */
    cleanup() {
        if (this.observer) {
            this.observer.disconnect();
            this.observer = null;
        }

        this.log('debug', '资源清理完成');
    }

    /**
     * 获取功能状态
     */
    getStatus() {
        return {
            isInitialized: this.isInitialized,
            isAIGeneratorPage: this.isAIGeneratorPage(),
            hasObserver: !!this.observer
        };
    }
}

// 创建全局实例
window.AyuAIGeneratorPage = new AyuAIGeneratorPage();

// 触发加载完成事件
document.dispatchEvent(new CustomEvent('ayuAIGeneratorPageLoaded', {
    detail: { loaded: true }
}));