/**
 * AYU引擎 - 内容脚本主入口
 * 专业的拼多多防比价工具核心模块
 * 
 * 功能特性:
 * - 智能页面类型检测和路由监听
 * - 模块化资源加载和功能初始化
 * - 商家后台批量处理功能
 * - 用户端商品详情页优化功能
 * - 统一的消息通信和错误处理
 */

class AyuContentScript {
    constructor() {
        this.initialized = false;
        this.loadStartTime = Date.now();
        this.activeFeatures = new Set();
        
        // 立即开始初始化
        this.init();
        
        // 调试：确保消息监听已设置
        console.log('[AYU ContentScript] 内容脚本已创建');
    }

    /**
     * 安全的日志输出方法
     */
    log(level, message, data = null) {
        if (typeof window.AyuLogger !== 'undefined') {
            if (level === 'system' && window.AyuLogger.system) {
                window.AyuLogger.system(message, data);
            } else if (window.AyuLogger[level]) {
                window.AyuLogger[level]('ContentScript', message, data);
            }
        } else {
            // 后备日志方案
            const prefix = '🎯 [AYU] [ContentScript]';
            const logMessage = `${prefix} ${message}`;
            data ? console.log(logMessage, data) : console.log(logMessage);
        }
    }

    /**
     * 初始化内容脚本
     */
    async init() {
        try {
            this.log('system', '内容脚本开始初始化', {
                url: window.location.href,
                title: document.title,
                timestamp: new Date().toLocaleTimeString()
            });

            // 等待页面检测器就绪
            await window.AyuPageDetector.waitForReady();
            
            // 设置全局扩展信息
            this.setupExtensionInfo();
            
            // 设置消息通信
            this.setupMessageCommunication();
            
            // 根据页面类型初始化功能
            await this.initializeFeatures();
            
            // 监听页面类型变化
            this.setupPageChangeListener();
            
            // 设置悬浮按钮清理机制
            this.setupFloatingButtonCleaner();
            
            this.initialized = true;
            const loadTime = Date.now() - this.loadStartTime;
            
            this.log('success', `初始化完成，用时 ${loadTime}ms`, {
                pageType: window.AyuPageDetector.getPageType(),
                features: Array.from(this.activeFeatures)
            });
            
        } catch (error) {
            this.log('error', '初始化失败', error);
        }
    }

    /**
     * 根据页面类型初始化功能模块
     */
    async initializeFeatures() {
        const pageType = window.AyuPageDetector.getPageType();
        
        this.log('info', '根据页面类型初始化功能', {
            pageType: pageType,
            url: window.location.href,
            hostname: window.location.hostname
        });
        
        // 先检查是否为AI制图页面（在iframe中）
        if (window.location.href.includes('aid.pinduoduo.com/design/ai-generator')) {
            this.log('info', '检测到AI制图页面，开始初始化页面优化功能');
            await this.initializeAIGeneratorPage();
            return;
        }

        switch (pageType) {
            case 'merchant_product_list':
                this.log('info', '开始初始化商家后台功能');
                await this.initializeMerchantBackend();
                break;
                
            case 'merchant_product_edit_ai':
                console.log('🎯 [Content] 检测到商品编辑AI页面，开始初始化');
                this.log('info', '开始初始化商品编辑AI制图功能');
                await this.initializeMerchantProductEditAI();
                break;
                
            case 'user_product_detail':
                this.log('info', '开始初始化用户端商品详情功能');
                await this.initializeProductDetail();
                break;
                
            default:
                this.log('info', `页面类型 ${pageType} 暂不支持特殊功能`, {
                    supportedTypes: ['merchant_product_list', 'merchant_product_edit_ai', 'user_product_detail']
                });
                break;
        }
    }

    /**
     * 初始化AI制图页面优化功能
     */
    async initializeAIGeneratorPage() {
        try {
            this.log('info', '开始初始化AI制图页面优化功能');

            // 加载AI制图页面功能模块
            const moduleUrl = chrome.runtime.getURL('js/features/ai-generator-page.js');
            
            // 动态注入脚本
            const script = document.createElement('script');
            script.src = moduleUrl;
            script.async = true;
            
            // 等待脚本加载完成
            await new Promise((resolve, reject) => {
                script.onload = () => {
                    this.log('success', 'AI制图页面优化模块加载完成');
                    this.activeFeatures.add('ai_generator_page');
                    resolve();
                };
                script.onerror = (error) => {
                    this.log('error', 'AI制图页面优化模块加载失败', error);
                    reject(error);
                };
                
                document.head.appendChild(script);
            });

        } catch (error) {
            this.log('error', '初始化AI制图页面优化功能失败', error);
        }
    }

    /**
     * 初始化商家后台批量处理功能
     */
    async initializeMerchantBackend() {
        try {
            this.log('info', '初始化商家后台功能');
            
            // 检查当前URL是否为指定的商家后台页面
            const currentUrl = window.location.href;
            const isTargetPage = currentUrl.includes('mms.pinduoduo.com/goods/goods_list');
            
            if (!isTargetPage) {
                this.log('info', '非目标商家后台页面，跳过初始化', {
                    currentUrl: currentUrl,
                    targetPattern: 'mms.pinduoduo.com/goods/goods_list'
                });
                return;
            }
            
            // 向服务工作器发送商家后台启动消息，这会触发远程脚本注入
            this.log('debug', '准备发送商家后台启动命令到service-worker');
            
            try {
                const startTime = Date.now();
                const response = await this.sendMessageToBackground({ cmd: "startMerchant" });
                const responseTime = Date.now() - startTime;
                
                this.log('debug', '商家后台启动命令响应', {
                    response: response,
                    responseTime: responseTime + 'ms'
                });
                
                // 等待脚本注入和执行
                this.log('debug', '等待商家后台脚本注入完成...');
                
                // 分阶段检查商家后台工具栏
                const checkMerchantToolbar = (attempt) => {
                    setTimeout(() => {
                        const toolbar = document.querySelector('.ayu-merchant-toolbar');
                        this.log('debug', `第${attempt}次检查商家后台工具栏`, {
                            found: !!toolbar,
                            element: toolbar
                        });
                        
                        if (toolbar) {
                            this.log('success', '商家后台工具栏已成功显示');
                        } else if (attempt < 5) {
                            checkMerchantToolbar(attempt + 1);
                        } else {
                            this.log('warn', '商家后台工具栏未显示，检查脚本注入情况', {
                                scripts: Array.from(document.querySelectorAll('script[src*="merchant"]')),
                                allScripts: Array.from(document.querySelectorAll('script')).map(s => s.src).filter(s => s.includes('ayu') || s.includes('fbj'))
                            });
                        }
                    }, attempt * 1000);
                };
                
                checkMerchantToolbar(1);
                
            } catch (error) {
                this.log('error', '发送商家后台启动命令失败', {
                    error: error,
                    message: error.message,
                    stack: error.stack
                });
                
                // 检查chrome.runtime是否可用
                this.log('debug', 'Chrome runtime状态检查', {
                    runtimeAvailable: !!chrome.runtime,
                    runtimeId: chrome.runtime?.id,
                    lastError: chrome.runtime?.lastError
                });
            }
            
            this.activeFeatures.add('merchant_backend');
            this.log('success', '商家后台功能初始化完成');
            
        } catch (error) {
            this.log('error', '商家后台功能初始化失败', error);
        }
    }

    /**
     * 初始化商品编辑AI制图功能
     */
    async initializeMerchantProductEditAI() {
        try {
            this.log('info', '初始化商品编辑AI制图功能');
            
            // 简化检查，页面类型检测已经验证了URL和参数
            const currentUrl = window.location.href;
            console.log('🎯 [Content] AI制图页面初始化开始', { currentUrl });
            
            // 向服务工作器发送AI制图功能启动消息
            this.log('debug', '准备发送AI制图功能启动命令到service-worker');
            
            try {
                const startTime = Date.now();
                const urlParams = new URLSearchParams(window.location.search);
                const id = urlParams.get('id');
                const goods_id = urlParams.get('goods_id');
                
                console.log('🎯 [Content] 发送启动命令', { id, goods_id });
                
                const response = await this.sendMessageToBackground({ 
                    cmd: "startMerchantProductEditAI",
                    data: { id, goods_id }
                });
                const responseTime = Date.now() - startTime;
                
                this.log('debug', 'AI制图功能启动命令响应', {
                    response: response,
                    responseTime: responseTime + 'ms'
                });
                
                // 等待脚本注入和执行
                this.log('debug', '等待AI制图功能脚本注入完成...');
                
                // 分阶段检查AI制图悬浮按钮
                const checkAIFloatingButton = (attempt) => {
                    setTimeout(() => {
                        const aiButton = document.getElementById('ayu-ai-drawing-button');
                        this.log('debug', `第${attempt}次检查AI制图悬浮按钮`, {
                            found: !!aiButton,
                            element: aiButton
                        });
                        
                        if (aiButton) {
                            this.log('success', 'AI制图悬浮按钮已成功显示');
                        } else if (attempt < 5) {
                            checkAIFloatingButton(attempt + 1);
                        } else {
                            this.log('warn', 'AI制图悬浮按钮未显示，检查脚本注入情况', {
                                scripts: Array.from(document.querySelectorAll('script[src*="merchant-product-edit"]')),
                                allScripts: Array.from(document.querySelectorAll('script')).map(s => s.src).filter(s => s.includes('ayu') || s.includes('fbj'))
                            });
                        }
                    }, attempt * 1000);
                };
                
                checkAIFloatingButton(1);
                
            } catch (error) {
                this.log('error', '发送AI制图功能启动命令失败', {
                    error: error,
                    message: error.message,
                    stack: error.stack
                });
                
                // 检查chrome.runtime是否可用
                this.log('debug', 'Chrome runtime状态检查', {
                    runtimeAvailable: !!chrome.runtime,
                    runtimeId: chrome.runtime?.id,
                    lastError: chrome.runtime?.lastError
                });
            }
            
            this.activeFeatures.add('merchant_product_edit_ai');
            this.log('success', '商品编辑AI制图功能初始化完成');
            
        } catch (error) {
            this.log('error', '商品编辑AI制图功能初始化失败', error);
        }
    }

    /**
     * 初始化商品详情页功能
     */
    async initializeProductDetail() {
        try {
            this.log('info', '初始化商品详情页功能');
            
            // 检查当前URL是否为指定的用户页面
            const currentUrl = window.location.href;
            const isTargetPage = currentUrl.includes('yangkeduo.com/goods.html');

            if (!isTargetPage) {
                this.log('info', '非目标用户页面，跳过初始化', {
                    currentUrl: currentUrl,
                    targetPattern: 'yangkeduo.com/goods.html'
                });
                return;
            }
            
            // 向服务工作器发送启动消息，这会触发浮动窗口脚本注入
            this.log('debug', '准备发送启动命令到service-worker');
            
            try {
                const startTime = Date.now();
                const response = await this.sendMessageToBackground({ cmd: "start" });
                const responseTime = Date.now() - startTime;
                
                this.log('debug', '启动命令响应', {
                    response: response,
                    responseTime: responseTime + 'ms'
                });
                
                // 等待脚本注入和执行
                this.log('debug', '等待脚本注入完成...');
                
                // 分阶段检查浮动窗口，但不检查悬浮批量按钮
                const checkFloatingWindow = (attempt) => {
                    setTimeout(() => {
                        const floatingBar = document.getElementById('ayu-floating-bar');
                        this.log('debug', `第${attempt}次检查浮动窗口`, {
                            found: !!floatingBar,
                            element: floatingBar
                        });
                        
                        // 确保移除悬浮批量按钮（如果存在）
                        const batchButton = document.getElementById('ayu-floating-batch-button');
                        if (batchButton) {
                            batchButton.remove();
                            this.log('info', '已移除悬浮批量按钮');
                        }
                        
                        if (floatingBar) {
                            this.log('success', '浮动窗口已成功显示');
                        } else if (attempt < 5) {
                            checkFloatingWindow(attempt + 1);
                        } else {
                            this.log('warn', '浮动窗口未显示，检查脚本注入情况', {
                                scripts: Array.from(document.querySelectorAll('script[src*="floatingBar"]')),
                                allScripts: Array.from(document.querySelectorAll('script')).map(s => s.src).filter(s => s.includes('ayu') || s.includes('fbj'))
                            });
                        }
                    }, attempt * 1000);
                };
                
                checkFloatingWindow(1);
                
            } catch (error) {
                this.log('error', '发送启动命令失败', {
                    error: error,
                    message: error.message,
                    stack: error.stack
                });
                
                // 检查chrome.runtime是否可用
                this.log('debug', 'Chrome runtime状态检查', {
                    runtimeAvailable: !!chrome.runtime,
                    runtimeId: chrome.runtime?.id,
                    lastError: chrome.runtime?.lastError
                });
            }
            
            this.activeFeatures.add('product_detail');
            this.log('success', '商品详情页功能初始化完成');
            
        } catch (error) {
            this.log('error', '商品详情页功能初始化失败', error);
        }
    }

    /**
     * 动态加载功能模块
     */
    async loadFeatureModules(modules) {
        const loadPromises = modules.map(module => {
            if (module.endsWith('.css')) {
                return this.loadCSS(chrome.runtime.getURL(module));
            } else if (module.endsWith('.js')) {
                return this.loadScript(chrome.runtime.getURL(module));
            }
        });
        
        await Promise.all(loadPromises);
        this.log('debug', `模块加载完成`, modules);
    }

    /**
     * 动态加载远程模块
     */
    async loadRemoteModules(modules) {
        const loadPromises = modules.map(module => {
            if (module.type === 'css') {
                return this.loadCSS(module.url);
            } else if (module.type === 'js') {
                return this.loadScript(module.url);
            }
        });
        
        await Promise.all(loadPromises);
        this.log('debug', `远程模块加载完成`, modules);
    }

    /**
     * 等待模块加载并可用
     */
    async waitForModule(moduleName, timeout = 5000) {
        return new Promise((resolve, reject) => {
            const startTime = Date.now();
            let checkCount = 0;
            
            // 也监听自定义事件
            const handleCustomEvent = (event) => {
                if (event.detail && event.detail.loaded) {
                    this.log('debug', `模块 ${moduleName} 通过事件加载完成`);
                    document.removeEventListener('ayuMerchantBackendLoaded', handleCustomEvent);
                    resolve();
                }
            };
            
            if (moduleName === 'AyuMerchantBackend') {
                document.addEventListener('ayuMerchantBackendLoaded', handleCustomEvent);
            }
            
            const checkModule = () => {
                checkCount++;
                this.log('debug', `检查模块 ${moduleName} (第${checkCount}次)`, {
                    exists: !!window[moduleName],
                    elapsed: Date.now() - startTime
                });
                
                if (window[moduleName]) {
                    this.log('debug', `模块 ${moduleName} 已就绪`);
                    document.removeEventListener('ayuMerchantBackendLoaded', handleCustomEvent);
                    resolve();
                } else if (Date.now() - startTime > timeout) {
                    this.log('error', `模块 ${moduleName} 加载超时，检查了${checkCount}次`, {
                        windowKeys: Object.keys(window).filter(k => k.includes('Ayu')),
                        scripts: Array.from(document.querySelectorAll('script')).map(s => s.src).filter(s => s.includes('merchant'))
                    });
                    document.removeEventListener('ayuMerchantBackendLoaded', handleCustomEvent);
                    reject(new Error(`模块 ${moduleName} 加载超时`));
                } else {
                    setTimeout(checkModule, 50);
                }
            };
            
            checkModule();
        });
    }

    /**
     * 加载CSS文件
     */
    loadCSS(url) {
        return new Promise((resolve, reject) => {
            // 检查是否已经加载
            if (document.querySelector(`link[href="${url}"]`)) {
                resolve();
                return;
            }

            const link = document.createElement('link');
            link.rel = 'stylesheet';
            link.type = 'text/css';
            link.href = url;
            
            link.onload = () => resolve();
            link.onerror = () => reject(new Error(`Failed to load CSS: ${url}`));
            
            document.head.appendChild(link);
        });
    }

    /**
     * 加载JavaScript文件
     */
    loadScript(url) {
        return new Promise((resolve, reject) => {
            // 检查是否已经加载
            if (document.querySelector(`script[src="${url}"]`)) {
                resolve();
                return;
            }

            const script = document.createElement('script');
            script.src = url;
            script.type = 'text/javascript';
            
            script.onload = () => {
                // 给脚本执行一些时间
                setTimeout(resolve, 100);
            };
            script.onerror = () => reject(new Error(`Failed to load script: ${url}`));
            
            document.head.appendChild(script);
        });
    }

    /**
     * 设置消息通信
     */
    setupMessageCommunication() {
        console.log('[AYU ContentScript] 开始设置消息通信');
        
        // 监听来自扩展后台的消息
        chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
            console.log('[AYU ContentScript] 收到后台消息:', message);
            this.handleBackgroundMessage(message, sender, sendResponse);
            return true; // 保持异步响应通道开放
        });

        // 监听页面内消息
        /* window.addEventListener('message', (event) => {
            if (event.source !== window) return;
            console.log('[AYU ContentScript] 收到页面消息:', event.data);
            this.handlePageMessage(event);
        });
 */
        console.log('[AYU ContentScript] 消息通信设置完成');
        this.log('debug', '消息通信设置完成');
    }

    /**
     * 处理来自后台的消息
     */
    handleBackgroundMessage(message, sender, sendResponse) {
        // 转发消息到页面
        window.postMessage({
            ...message,
            source: 'ayu_extension'
        }, "*");
    }

    /**
     * 处理页面内消息
     */
    /* handlePageMessage(event) {
        const { data } = event;
        
        // 处理商家端的API请求
        if (data && data.type === 'AYU_API_REQUEST' && data.source === 'merchant_backend') {
            console.log('[AYU ContentScript] 收到商家端API请求:', data);
            this.handleMerchantAPIRequest(data);
            return;
        }
        
        // 处理需要转发到后台的消息
        if (data && data.forwardToBackground) {
            this.sendMessageToBackground(data).then(response => {
                window.postMessage({
                    type: 'ayu_extension_response',
                    requestId: data.requestId,
                    response: response
                }, '*');
            }).catch(error => {
                window.postMessage({
                    type: 'ayu_extension_error',
                    requestId: data.requestId,
                    error: error.message
                }, '*');
            });
        }
    } */

    /**
     * 处理商家端API请求
     */
    /* async handleMerchantAPIRequest(message) {
        const { action, data, messageId } = message;
        
        console.log(`[AYU ContentScript] 处理API请求: ${action}`, { messageId, data });
        
        try {
            let response;
            
            switch (action) {
                case 'SEND_CODE':
                    console.log('[AYU ContentScript] 处理发送验证码请求');
                    response = await this.sendCodeRequest(data);
                    break;
                case 'USER_LOGIN':
                    console.log('[AYU ContentScript] 处理用户登录请求');
                    response = await this.userLoginRequest(data);
                    break;
                case 'GET_USER_INFO':
                    console.log('[AYU ContentScript] 处理获取用户信息请求');
                    response = await this.getUserInfoRequest(data);
                    break;
                case 'HTTP_POST':
                    console.log('[AYU ContentScript] 处理通用HTTP POST请求');
                    response = await this.httpPostRequest(data);
                    break;
                default:
                    throw new Error(`未知的API操作: ${action}`);
            }
            
            console.log(`[AYU ContentScript] API请求成功: ${action}`, response);
            
            // 发送响应
            const responseMessage = {
                type: 'AYU_API_RESPONSE',
                messageId: messageId,
                success: true,
                data: response
            };
            
            console.log('[AYU ContentScript] 发送成功响应:', responseMessage);
            window.postMessage(responseMessage, '*');
            
        } catch (error) {
            console.error(`[AYU ContentScript] API请求处理失败: ${action}`, error);
            
            // 发送错误响应
            const errorMessage = {
                type: 'AYU_API_RESPONSE',
                messageId: messageId,
                success: false,
                error: error.message
            };
            
            console.log('[AYU ContentScript] 发送错误响应:', errorMessage);
            window.postMessage(errorMessage, '*');
        }
    } */

    /**
     * 发送验证码请求
     */
    /* async sendCodeRequest(data) {
        const response = await fetch('https://ayu.wanwutime.com/api/user/sendCode', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify(data)
        });
        return response.json();
    } */

    /**
     * 用户登录请求
     */
    /* async userLoginRequest(data) {
        const response = await fetch('https://ayu.wanwutime.com/api/user/login', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify(data)
        });
        return response.json();
    }*/

    /**
     * 获取用户信息请求
     */
    /* async getUserInfoRequest(data) {
        const response = await fetch('https://ayu.wanwutime.com/api/user/info', {
            method: 'GET',
            headers: {
                'Authorization': data.token,
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            }
        });
        return response.json();
    } */

    /**
     * 通用HTTP POST请求
     */
    /* async httpPostRequest(data) {
        console.log('[AYU ContentScript] 发起HTTP POST请求:', data.url);
        
        const response = await fetch(data.url, {
            method: 'POST',
            headers: data.headers || {},
            body: data.data
        });
        
        console.log('[AYU ContentScript] HTTP POST请求响应状态:', response.status);
        
        if (!response.ok) {
            throw new Error(`HTTP请求失败: ${response.status} ${response.statusText}`);
        }
        
        // 尝试解析JSON，如果失败则返回文本
        try {
            const jsonData = await response.json();
            console.log('[AYU ContentScript] HTTP POST请求JSON响应:', jsonData);
            return jsonData;
        } catch (error) {
            const textData = await response.text();
            console.log('[AYU ContentScript] HTTP POST请求文本响应:', textData.substring(0, 200));
            return { text: textData };
        }
    } */

    /**
     * 向后台发送消息
     */
    sendMessageToBackground(message) {
        return new Promise((resolve, reject) => {
            chrome.runtime.sendMessage(message, (response) => {
                if (chrome.runtime.lastError) {
                    reject(new Error(chrome.runtime.lastError.message));
                } else {
                    resolve(response);
                }
            });
        });
    }

    /**
     * 设置扩展信息
     */
    setupExtensionInfo() {
        if (chrome.runtime.id) {
            localStorage.setItem("ayuPluginId", chrome.runtime.id);
            localStorage.setItem("ayuPluginName", window.AyuConfig.name);
            localStorage.setItem("ayuPluginVersion", window.AyuConfig.version);
        }
        
        this.log('debug', '扩展信息设置完成');
    }

    /**
     * 监听页面类型变化
     */
    setupPageChangeListener() {
        document.addEventListener('ayuPageTypeChanged', async (event) => {
            const { detail } = event;
            this.log('info', `页面类型变化: ${detail.oldType} -> ${detail.newType}`);
            
            // 清理旧功能
            this.cleanupFeatures();
            
            // 重新初始化功能
            await this.initializeFeatures();
        });
    }

    /**
     * 设置悬浮批量按钮清理机制
     */
    setupFloatingButtonCleaner() {
        // 立即清理一次
        this.cleanupFloatingBatchButton();
        
        // 定期清理悬浮批量按钮
        setInterval(() => {
            this.cleanupFloatingBatchButton();
        }, 2000); // 每2秒检查一次
        
        // 监听DOM变化，及时清理新添加的悬浮按钮
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                mutation.addedNodes.forEach((node) => {
                    if (node.nodeType === Node.ELEMENT_NODE) {
                        // 检查新添加的节点是否为悬浮批量按钮
                        if (node.id === 'ayu-floating-batch-button') {
                            node.remove();
                            this.log('info', '移除新添加的悬浮批量按钮');
                        }
                        // 检查新添加的节点内部是否包含悬浮批量按钮
                        const batchButton = node.querySelector && node.querySelector('#ayu-floating-batch-button');
                        if (batchButton) {
                            batchButton.remove();
                            this.log('info', '移除新添加节点内的悬浮批量按钮');
                        }
                    }
                });
            });
        });
        
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
        
        this.log('debug', '悬浮按钮清理机制已设置');
    }

    /**
     * 清理悬浮批量按钮
     */
    cleanupFloatingBatchButton() {
        const batchButton = document.getElementById('ayu-floating-batch-button');
        if (batchButton) {
            batchButton.remove();
            this.log('debug', '已清理悬浮批量按钮');
        }
    }

    /**
     * 清理功能模块
     */
    cleanupFeatures() {
        // 清理活跃功能
        this.activeFeatures.clear();
        
        // 可以在这里添加具体的清理逻辑
        this.log('debug', '功能模块清理完成');
    }

    /**
     * 获取扩展状态
     */
    getStatus() {
        return {
            initialized: this.initialized,
            pageType: window.AyuPageDetector.getPageType(),
            activeFeatures: Array.from(this.activeFeatures),
            loadTime: Date.now() - this.loadStartTime
        };
    }
}

// 等待依赖加载完成后初始化
function initializeWhenReady() {
    // 检查依赖是否加载完成
    if (typeof window.AyuLogger !== 'undefined' && 
        typeof window.AyuConfig !== 'undefined' && 
        typeof window.AyuPageDetector !== 'undefined') {
        
        // 创建内容脚本实例
        window.ayuContentScript = new AyuContentScript();
        
    } else {
        // 等待依赖加载
        setTimeout(initializeWhenReady, 50);
    }
}

// 页面加载完成后初始化
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeWhenReady);
} else {
    initializeWhenReady();
}
